/**********************************************************************/
/* Task           : Functions for direct access to                    */
/*                  serial port                                       */
/**********************************************************************/
#ifndef __SERUTIL_C                          /* Can also be #included */
#define __SERUTIL_C

#include <conio.h>
#include "8250.h"
#include "irqutil.h"


/**********************************************************************/
/* reset : Initialize serial port                                  */
/**------------------------------------------------------------------**/
/* Input  : iSerPort - base port of interface                         */
/*                     being initialized.                             */
/*          lBaud    - baud rate ( from 1 - 115200 )                  */
/*          bParams  - bit mask of remaining parameters               */
/*                     (s. SER_LCR_... bits)                          */
/* Output : TRUE  - port initialized successfully                     */
/*          FALSE - no port found                                     */
/**********************************************************************/
INT Uart::reset( INT  iSerPort, LONG lBaudRate, BYTE bParams )
{ WORD uDivisor;
  if( ser_UARTType( iSerPort ) != NOSER )
  {                                    /* Calculate baud rate divisor */
    uDivisor = ( WORD )( SER_MAXBAUD / lBaudRate );
    outp( iSerPort + SER_LINE_CONTROL,       /* Enable divisor access */
                inp( SER_LINE_CONTROL ) | SER_LCR_SETDIVISOR );
                                             /* Set baud rate divisor */
    outp( iSerPort + SER_DIVISOR_LSB, LOBYTE( uDivisor ) );
    outp( iSerPort + SER_DIVISOR_MSB, HIBYTE( uDivisor ) );
                                            /* Disable divisor access */
    outp( iSerPort + SER_LINE_CONTROL,
                inp( SER_LINE_CONTROL ) & ~SER_LCR_SETDIVISOR );

  /* Set other parameters only after resetting baud rate latch,       */
  /* because this operation clears all port parameters!               */
  /* Set transmission parameters other than baud rate                 */
    outp( iSerPort + SER_LINE_CONTROL, bParams );
                            /* Read a byte, to reverse possible error */
    inp( iSerPort + SER_TXBUFFER );
    return TRUE;
  }
  return FALSE;
}

/**********************************************************************/
/* ser_FIFOLevel : Set FIFO buffer size                               */
/**------------------------------------------------------------------**/
/* Input : 0 - FIFO buffer size = 0, disable and reset (1 byte)       */
/*         SER_FIFO_TRIGGER4/8/14 - size = 4, 8 or 14 bytes           */
/**********************************************************************/
VOID Uart::ser_FIFOLevel( INT iSerPort, BYTE bLevel )
{
  if( bLevel ) outp( iSerPort + SER_FIFO, bLevel | SER_FIFO_ENABLE );
  else         outp( iSerPort + SER_FIFO, SER_FIFO_RESETRECEIVE |
                                          SER_FIFO_RESETTRANSMIT );
}

/**********************************************************************/
/* ser_IsDataAvailable : Is data available to be read?                */
/**------------------------------------------------------------------**/
/* Input  : iSerPort - base port of interface being checked.          */
/* Output : == 0 : No byte available to be read                       */
/*          != 0 : Byte is available                                  */
/**------------------------------------------------------------------**/
/* Info : A byte is sent bit by bit, and becomes                      */
/*        a complete byte again only when the receiving port          */
/*        has combined the individual bits. This is what is           */
/*        being checked by this function.                             */
/**********************************************************************/
INT Uart::ser_IsDataAvailable( INT iSerPort )
{
  return inp( iSerPort + SER_LINE_STATUS ) & SER_LSR_DATARECEIVED;
}

/**********************************************************************/
/* ser_IsWritingPossible : Can port send next byte ?                  */
/**------------------------------------------------------------------**/
/* Input  : iSerPort - base port of interface being checked.          */
/* Output : == 0 : Byte cannot be sent.                               */
/*          != 0 : Port ready to send.                                */
/**------------------------------------------------------------------**/
/* Info : A serial port should not be used                            */
/*        to send a byte in the following cases:                      */
/*        1. A received byte has not yet been "retrieved"             */
/*           by the port.                                             */
/*        2. An old send request has not yet been completed.          */
/**********************************************************************/
INT Uart::ser_IsWritingPossible( INT iSerPort )
{
  return ( inp( iSerPort + SER_LINE_STATUS ) & SER_LSR_TSREMPTY );
}

/**********************************************************************/
/* set : Send a byte                                        */
/**------------------------------------------------------------------**/
/* Input : iSerPort - base port of interface through which            */
/*                    a byte is being sent.                           */
/*         bData    - byte to be sent                                 */
/*         uTimeOut - number of passes through loop                   */
/*                    after which a timeout error occurs              */
/*                    if the send was unsuccessful (if                */
/*                    iTimeOut = 0 the system waits "forever".)       */
/*         bSigMask - bit mask of signal lines being tested           */
/*                    (RTS, CTS, CD, RI)                              */
/*         bSigVals - signal line status after applying               */
/*                    above mask.                                     */
/* Output : == 0 - byte was sent                                      */
/*          != 0 - error                                              */
/**********************************************************************/
INT Uart::set( INT iSerPort, BYTE bData, UINT uTimeOut,
                   BYTE bSigMask, BYTE bSigVals )
{ if( uTimeOut )                                      /* Timeout loop */
  {
    while( !ser_IsWritingPossible( iSerPort ) && uTimeOut ) uTimeOut--;
    if( !uTimeOut ) return SER_ERRTIMEOUT;
  }
  else while( !ser_IsWritingPossible( iSerPort ) );          /* Wait! */

                                                 /* Test signal lines */
  if( ( ( BYTE ) inp( iSerPort + SER_MODEM_STATUS ) & bSigMask ) ==
                 bSigVals )
                {                     /* Pass byte being sent to port */
    outp( iSerPort + SER_TXBUFFER, bData );
                                                 /* Return port error */
    return inp( iSerPort + SER_LINE_STATUS ) & SER_LSR_ERRORMSK;
  }
  else return SER_ERRSIGNALS;
}

/**********************************************************************/
/* get : Receive byte                                        */
/**------------------------------------------------------------------**/
/* Input  : iSerPort - base port of interface through which           */
/*                     a byte is being received.                      */
/*          pData    - address of byte accepting                      */
/*                     received byte.                                 */
/*          uTimeOut - number of passes through loop                  */
/*                     after which a timeout error occurs             */
/*                     if the send was unsuccessful. (If              */
/*                     iTimeOut = 0 the system waits "forever".)      */
/*          bSigMask - bit mask of signal lines being tested          */
/*                     (RTS, CTS, CD, RI)                             */
/*          bSigVals - signal line status after applying              */
/*                    above mask.                                     */
/* Output : == 0 - byte was sent                                      */
/*          != 0 - error                                              */
/**********************************************************************/
INT Uart::get( INT iSerPort, PBYTE pData, UINT uTimeOut,
                  BYTE bSigMask, BYTE bSigVals )
{ if( uTimeOut )                                      /* Timeout loop */
  {
    while( !ser_IsDataAvailable( iSerPort ) && uTimeOut ) uTimeOut--;
    if( !uTimeOut ) return SER_ERRTIMEOUT;
  }
  else while( !ser_IsDataAvailable( iSerPort ) );            /* Wait! */

                                                 /* Test signal lines */
  if( ( ( BYTE ) inp( iSerPort + SER_MODEM_STATUS ) & bSigMask ) ==
          bSigVals )
  {                                     /* Read byte received by port */
    *pData = ( BYTE )inp( iSerPort + SER_RXBUFFER );
    return inp( iSerPort + SER_LINE_STATUS ) & SER_LSR_ERRORMSK;
  }
  else return SER_ERRSIGNALS;
}


/**********************************************************************/
/* ser_CLRIRQ : Disable serial interrupt requests to                  */
/*              IRQ controller.                                       */
/**------------------------------------------------------------------**/
/* Input : iSerPort - base port of interface that can no longer       */
/*                    issue IRQs to IRQ controller.                   */
/**********************************************************************/
VOID Uart::ser_CLRIRQ( INT iSerPort )
{
  outp( iSerPort + SER_MODEM_CONTROL,
        inp( iSerPort + SER_MODEM_CONTROL ) & ~SER_MCR_IRQENABLED );
}

/**********************************************************************/
/* ser_SETIRQ : Enable serial interrupt requests to                   */
/*              IRQ controller.                                       */
/**------------------------------------------------------------------**/
/* Input : iSerPort - base port of interface that must                */
/*                    issue IRQs to IRQ controller.                   */
/**********************************************************************/
VOID Uart::ser_SETIRQ( INT iSerPort )
{
  outp( iSerPort + SER_MODEM_CONTROL,
        inp( iSerPort + SER_MODEM_CONTROL ) | SER_MCR_IRQENABLED );
}

/**********************************************************************/
/* ser_SetIRQHandler : Set interrupt handler                          */
/**------------------------------------------------------------------**/
/* Input  : iSerPort  - base port of serial interface                 */
/*                      for which interrupt handler                   */
/*                      is being set.                                 */
/*          iSerIRQ   - IRQ(!) line reserved for port.                */
/*          lpHandler - address of interrupt handler.                 */
/*          bEnablers - conditions initiating                         */
/*                      an IRQ (see SER_IER_... bits)                 */
/* Output : Address of old IRQ handler                                */
/**********************************************************************/
VOID Uart::( _interrupt _FP *ser_SetIRQHandler( INT iSerPort,
                                           INT iSerIRQ,
                        VOID ( _interrupt _FP *lpHandler ) (),
                                          BYTE bEnablers ) ) ()
   {                                              /* Set IRQ enablers */
  outp( iSerPort + SER_IRQ_ENABLE, bEnablers );
  ser_SETIRQ( iSerPort );             /* Issue IRQs to IRQ controller */
                              /* Set handler (IRQ is "enabled" there) */
  return irq_SetHandler( iSerIRQ, lpHandler );
}

/**********************************************************************/
/* ser_RestoreIRQHandler : Restore old IRQ handler                    */
/**------------------------------------------------------------------**/
/* Input  : iSerPort  - base port of serial interface whose           */
/*                      old interrupt handler is being restored       */
/*          iSerIRQ   - IRQ line reserved for port.                   */
/*          lpHandler - address of old interrupt handler .            */
/**********************************************************************/
VOID Uart::ser_RestoreIRQHandler( INT iSerPort,
                            INT iSerIRQ,
         VOID ( _interrupt _FP *lpHandler ) () )
{
  ser_CLRIRQ( iSerPort );           /* No more IRQs to IRQ controller */
                              /* Set handler and clear all "enablers" */
  ser_SetIRQHandler( iSerPort, iSerIRQ, lpHandler, 0 );
  irq_Disable( iSerIRQ );      /* Also disable IRQs by the controller */
}

/**********************************************************************/
/* ser_PrintError : Output error message                              */
/**------------------------------------------------------------------**/
/* Input  : pWin - address of window where output should appear       */
/*                 or ZERO for output to screen.                      */
/*             e - error                                              */
/**********************************************************************/
VOID Uart::ser_PrintError( PWINDOW pWin, INT e )
{
  switch( e )
  {
    case SER_LSR_DATARECEIVED:
         win_printf( pWin, "Old data!\n" );
    break;
    case SER_ERRTIMEOUT:
         win_printf( pWin, "Timeout error! ");
    break;
    case SER_ERRSIGNALS:
         win_printf( pWin, "Signal lines!");
    break;
    default:
      if( e & SER_LSR_OVERRUNERROR )
          win_printf( pWin, "Overrun error!\n" );
      if( e & SER_LSR_PARITYERROR )
          win_printf( pWin, "Parity error!\n" );
      if( e & SER_LSR_FRAMINGERROR )
          win_printf( pWin, "Framing error!\n" );
      if( e & SER_LSR_BREAKDETECT )
          win_printf( pWin, "Break detect!\n" );
  }
}

/**********************************************************************/
/* ser_GetBaud : Get current baud rate for port                       */
/**------------------------------------------------------------------**/
/* Input  : iSerPort - Base address of port whose                     */
/*                     baud rate is being determined.                 */
/* Output : baud rate                                                 */
/**********************************************************************/
LONG Uart::ser_GetBaud( INT iSerPort )
{ UINT uDivisor;
  BYTE bSettings;

  _disable();
  bSettings = ( BYTE ) inp( iSerPort + SER_LINE_CONTROL );
  outp( iSerPort + SER_LINE_CONTROL, bSettings | SER_LCR_SETDIVISOR );
                                            /* Read baud rate divisor */
  uDivisor = MAKEWORD( inp( iSerPort + SER_DIVISOR_MSB ),
                       inp( iSerPort + SER_DIVISOR_LSB ) );

  outp( iSerPort + SER_LINE_CONTROL, bSettings );
  _enable();
  if( uDivisor ) return SER_MAXBAUD / uDivisor;
  return 0L;
}

#endiF
