#ifndef __ix86_h
#define __ix86_h

#include <system/setup.h>
#include <cpu.h>
#include "common.h"

#define ASM __asm__
#define ASMV __asm__ __volatile__

__BEGIN_SYS

class Ix86: private CPU_Common
{
public:
    static CPU_Reg64 get_tsc()
	{CPU_Reg64 tsc; ASMV("rdtsc" : "=A" (tsc) : ); return tsc;}

    static CPU_Reg64 get_time_stamp()
	{return get_tsc();}
    
    static CPU_Reg32 htonl(CPU_Reg32 v)
	{ASMV("bswap %0" : "=r" (v) : "0" (v), "r" (v)); return v;}
    static CPU_Reg16 htons(CPU_Reg16 v)
	{return ((v << 8) & 0xFF00) | ((v >> 8) & 0x00FF);}
    static CPU_Reg32 ntohl(CPU_Reg32 v)
	{ASMV("bswap %0" : "=r" (v) : "0" (v), "r" (v)); return v;}
    static CPU_Reg16 ntohs(CPU_Reg16 v)
	{return ((v << 8) & 0xFF00) | ((v >> 8) & 0x00FF);}

    static CPU_Reg16 adjust_selector(CPU_Reg16 selector, int rpl)
	{return ((selector) << 3) | (rpl);}

    static void switch_tss(CPU_Reg32 tss_selector) {
	struct {CPU_Reg32 offset; CPU_Reg32 selector;} address;
	address.offset   = 0; address.selector = tss_selector;
	ASMV("ljmp %0" : "=o" (address));
    }

    void switch_context(CPU_Reg16 context) const
	{switch_tss(adjust_selector(GDT_TSS + context, 0));}

    static CPU_Reg16 get_tr()
	{CPU_Reg16 tr; ASMV("str %0" : "=r"(tr) :); return tr;}
    static CPU_Reg32 get_cr0()
	{CPU_Reg32 cr0; ASMV("movl %%cr0,%0" : "=r"(cr0) :); return cr0;}
    static CPU_Reg32 get_cr2()
	{CPU_Reg32 cr2; ASMV("movl %%cr2,%0" : "=r"(cr2) :); return cr2;}
    static CPU_Reg32 get_cr3()
	{CPU_Reg32 cr3; ASMV("movl %%cr3,%0" : "=r"(cr3) :); return cr3;}
    static void get_gdtr(CPU_Reg16 *limit, CPU_Reg32 *base) {
	char aux[6];
	ASMV("sgdt %0" : "=m"(aux[0]) :);
	*limit = *((CPU_Reg16 *)&aux[0]);
	*base = *((CPU_Reg32 *)&aux[2]);
    }
    static void get_idtr(CPU_Reg16 *limit, CPU_Reg32 *base) {
	char aux[6];
	ASMV("sidt %0" : "=m"(aux[0]) :);
	*limit = *((CPU_Reg16 *)&aux[0]);
	*base = *((CPU_Reg32 *)&aux[2]);
    }
    static CPU_Reg32 get_eflags() {
	CPU_Reg32 eflags;
	ASM("pushfl");
	ASMV("popl %0" : "=r"(eflags) :);
	return eflags;
    }

    static void sti() {ASM("sti");}
    static void set_tr(CPU_Reg16 tr)
	{ASMV("ltr %0" : : "r"(tr));}
    static void set_cr0(CPU_Reg32 cr0)
	{ASMV("movl %0,%%cr0" : : "r"(cr0));}
    static void set_cr3(CPU_Reg32 cr3)
	{ASMV("movl %0,%%cr3" : : "r"(cr3));}
    static void set_gdtr(CPU_Reg16 limit, CPU_Reg32 base) {
	char aux[6];
	*((CPU_Reg16 *)&aux[0]) = limit;
	*((CPU_Reg32 *)&aux[2]) = base;
	ASMV("lgdt %0" : : "m" (aux[0]));
    }
    static void set_idtr(CPU_Reg16 limit, CPU_Reg32 base) {
	char aux[6];
	*((CPU_Reg16 *)&aux[0]) = limit;
	*((CPU_Reg32 *)&aux[2]) = base;
	ASMV("lidt %0" : : "m" (aux[0]));
    }
    static void set_eflags(CPU_Reg32 eflags) {
	ASM("pushl %eax");
	ASM("popfl");
    }

    static CPU_Reg8 inb(CPU_Reg16 port) {
	CPU_Reg8 value;
	ASMV("inb %1,%0" : "=a"(value) : "d"(port) : "al", "dx");
	return value;
    }
    static CPU_Reg16 inw(CPU_Reg16 port) {
	CPU_Reg16 value;
	ASMV("inw %1,%0" : "=a"(value) : "d"(port) : "ax", "dx");
	return value;
    }

    static void outb(CPU_Reg16 port, CPU_Reg8 value)
	{ASMV("outb %1,%0" : : "d"(port), "a"(value) : "dx", "al");}
    static void outw(CPU_Reg16 port, CPU_Reg16 value)
	{ASMV("outw %1,%0" : : "d"(port), "a"(value) : "dx", "ax");}
    
    static void flush_tlb() {
	ASM("movl %cr3,%eax");
	ASM("movl %eax,%cr3");
    }

    static void invlpg(CPU_Reg32 logical_address)
	{ASMV("invlpg %0" : : "m"(logical_address));}
    
    static void push_all() {ASM("pushal");}
    static void pop_all() {ASM("popal");}
  
    static void init_fpu() {ASM("fninit");}

    static int init(System_Info *si);

private:
    // data
};

inline unsigned short htons(unsigned short value)
  { return Ix86::htons(value); }
inline unsigned long htonl(unsigned long value)
  { return Ix86::htonl(value); }
inline unsigned short ntohs(unsigned short value)
  { return Ix86::ntohs(value); }
inline unsigned long ntohl(unsigned long value)
  { return Ix86::ntohl(value); }

__END_SYS

#endif

