/*======================================================================*/
/* INTEL x86 SUPPORT FUNCTIONS						*/
/*                                                                      */
/* Author: Guto								*/
/*======================================================================*/
#include <pc/ix86.h>

#define ASM __asm__
#define ASMV __asm__ __volatile__

void ix86_sti(void)
{
  ASM("sti");
}

void ix86_set_tr(Ix86_Reg16 tr)
{
  ASMV("ltr %0" : : "r"(tr));
}

Ix86_Reg16 ix86_get_tr(void)
{
  Ix86_Reg16 tr;

  ASMV("str %0" : "=r"(tr) :);
  return tr;
}

void ix86_set_cr0(Ix86_Reg32 cr0)
{
  ASMV("movl %0,%%cr0" : : "r"(cr0));
}

Ix86_Reg32 ix86_get_cr0(void)
{
  Ix86_Reg32 cr0;

  ASMV("movl %%cr0,%0" : "=r"(cr0) :);
  return cr0;
}

Ix86_Reg32 ix86_get_cr2(void)
{
  Ix86_Reg32 cr2;

  ASMV("movl %%cr2,%0" : "=r"(cr2) :);
  return cr2;
}

void ix86_set_cr3(Ix86_Reg32 cr3)
{
  ASMV("movl %0,%%cr3" : : "r"(cr3));
}

Ix86_Reg32 ix86_get_cr3(void)
{
  Ix86_Reg32 cr3;

  ASMV("movl %%cr3,%0" : "=r"(cr3) :);
  return cr3;
}

void ix86_set_gdtr(Ix86_Reg16 limit, Ix86_Reg32 base)
{
  char aux[6];

  *((Ix86_Reg16 *)&aux[0]) = limit;
  *((Ix86_Reg32 *)&aux[2]) = base;
  ASMV("lgdt %0" : : "m" (aux[0]));

/*
  ASMV("movw 0x8(%ebp),%ax");
  ASMV("movw %ax,0xa(%ebp)");
  ASMV("lgdt 0xa(%ebp)");
*/
}

void ix86_get_gdtr(Ix86_Reg16 *limit, Ix86_Reg32 *base)
{
  char aux[6];

  ASMV("sgdt %0" : "=m"(aux[0]) :);
  *limit = *((Ix86_Reg16 *)&aux[0]);
  *base = *((Ix86_Reg32 *)&aux[2]);
}

void ix86_set_idtr(Ix86_Reg16 limit, Ix86_Reg32 base) /* PROBLEMS */
{
  char aux[6];

  *((Ix86_Reg16 *)&aux[0]) = limit;
  *((Ix86_Reg32 *)&aux[2]) = base;
  ASMV("lidt %0" : : "m" (aux[0]));

  /*
  ASMV("movw 0x8(%ebp),%ax");
  ASMV("movw %ax,0xa(%ebp)");
  ASMV("lidt 0xa(%ebp)");
  */
}

void ix86_get_idtr(Ix86_Reg16 *limit, Ix86_Reg32 *base) /* PROBLEMS */
{
  char aux[6];

  ASMV("sidt %0" : "=m"(aux[0]) :);
  *limit = *((Ix86_Reg16 *)&aux[0]);
  *base = *((Ix86_Reg32 *)&aux[2]);
}

void ix86_set_eflags(Ix86_Reg32 eflags)
{
  ASM("pushl %eax");
  ASM("popfl");
}

Ix86_Reg32 ix86_get_eflags(void)
{
  Ix86_Reg32 eflags;

  ASM("pushfl");
  ASMV("popl %0" : "=r"(eflags) :);
  return eflags;
}

Ix86_Reg64 ix86_get_tsc(void)
{
  Ix86_Reg64 tsc;

  ASMV("rdtsc" : "=a" (*((Ix86_Reg32 *)&tsc)),
      "=d" (*(((Ix86_Reg32 *)&tsc) + 1)) :);
  return tsc;
}

Ix86_Reg8 ix86_inb(Ix86_Reg16 port)
{
  Ix86_Reg8 value;

  ASMV("inb %1,%0" : "=a"(value) : "d"(port));
/*   ASMV("inb %1,%0" : "=a"(value) : "d"(port) : "al", "dx"); */
  return value;
}

void ix86_outb(Ix86_Reg16 port, Ix86_Reg8 value)
{
  ASMV("outb %1,%0" : : "d"(port), "a"(value));
/*  ASMV("outb %1,%0" : : "d"(port), "a"(value) : "dx", "al"); */
}

Ix86_Reg16 ix86_inw(Ix86_Reg16 port)
{
  Ix86_Reg16 value;

  ASMV("inw %1,%0" : "=a"(value) : "d"(port));
/*   ASMV("inw %1,%0" : "=a"(value) : "d"(port) : "ax", "dx"); */
  return value;
}

void ix86_outw(Ix86_Reg16 port, Ix86_Reg16 value)
{
  ASMV("outw %1,%0" : : "d"(port), "a"(value));
/*   ASMV("outw %1,%0" : : "d"(port), "a"(value) : "dx", "ax"); */
}

void ix86_flush_tlb(void)
{
  ASM("movl %cr3,%eax");
  ASM("movl %eax,%cr3");
}

void ix86_invlpg(Ix86_Reg32 logical_address)
{
  ASMV("invlpg %0" : : "m"(logical_address));
}

/*======================================================================*/
/* ix86_set_gdt_entry                                                   */
/*                                                                      */
/* Desc: Set an entry in GDT. This function cracks the parameters,      */
/*       so that we don't go crazy moving bits around.                  */
/*                                                                      */
/* Parm: gdt     -> GDT base address                                    */
/*       index   -> index into GDT                                      */
/*       base    -> 32-bit base of the new segment                      */
/*       limit   -> 16-bit limit                                        */
/*       flags   -> flags (present, dpl, sys/app, type)                 */
/*======================================================================*/
void ix86_set_gdt_entry(Ix86_GDT_Desc *gdt, int index, Ix86_Reg32 base,
			Ix86_Reg32 limit, Ix86_Reg8 flags)
{
  int flags2;

  if(flags & IX86_SEG_NOSYS)
    flags2 = IX86_SEG_4K | IX86_SEG_32;
  else
    flags2 = 0;    

  gdt[index].limit_15_00 = (Ix86_Reg16) limit;
  gdt[index].base_15_00 = (Ix86_Reg16) base;
  gdt[index].base_23_16 = (Ix86_Reg8) (base >> 16);
  gdt[index].p_dpl_s_type = flags;
  gdt[index].g_d_0_a_limit_19_16 = flags2 | ((Ix86_Reg8) (limit >> 16));
  gdt[index].base_31_24 = (Ix86_Reg8) (base >> 24);
}

/*======================================================================*/
/* ix86_set_idt_entry                                                   */
/*                                                                      */
/* Desc: Set an entry in IDT, cracking parameters as necessary.         */
/*                                                                      */
/* Parm: idt       -> IDT base address                                  */
/*       index     -> entry no.                                         */
/*       selector  -> 13-bit segment selector (TI and RPL will be set)  */
/*       offset    -> 32-bit offset                                     */
/*       flags     -> 16-bit flags (P,DPL,S,C,E,W,A)	                */
/*======================================================================*/
void ix86_set_idt_entry(Ix86_IDT_Desc *idt, int index, Ix86_Reg16 selector,
                     Ix86_Reg32 offset, Ix86_Reg16 flags)
{
  idt[index].offset_15_00 = (Ix86_Reg16) offset;
  idt[index].selector = selector << 3;
  idt[index].zero = 0;
  idt[index].p_dpl_0_d_1_1_0 = flags;
  idt[index].offset_31_16 = (Ix86_Reg16) (offset >> 16);
}

void ix86_switch_tss(Ix86_Reg32 tss_selector)
{
  struct
  {
    Ix86_Reg32 offset;
    Ix86_Reg32 selector;
  } address;

  address.offset   = 0;
  address.selector = tss_selector;

  ASMV("ljmp *%0" : "=o" (address));
}

void ix86_push_all()
{
  ASM("pushal");
}

void ix86_pop_all()
{
  ASM("popal");
}

void ix86_init_fpu()
{
  ASM("fninit");
}
