/*======================================================================*/
/* IMPLEMENTATION OF A SIMPLE OS LEVEL PRINTF                           */
/*                                                                      */
/* Auth: Guto                                                           */
/*======================================================================*/
#include <stdarg.h>
#include <pc/cga.h>
#include <system/kprintf.h>

/*======================================================================*/
/* GLOBALS	                                                        */
/*======================================================================*/
static const char dec_digits[] = "0123456789";
static const char hex_digits[] = "0123456789abcdef";

/*======================================================================*/
/* kprintf                                                              */
/*									*/
/* Desc: prints a formated string using the direct video access		*/
/*	 functions provided by CGA. Valid formats are %c, %s, %d, %x	*/
/*	 and %p.							*/
/*									*/
/* Parm: fmt -> the format string					*/
/*	 ... -> arguments to be printed according to the format string	*/
/*======================================================================*/
void kprintf(const char *fmt, ...)
{
  va_list arg_list;
  char buf[64];
  int i;

  va_start(arg_list, fmt);

  for(i = 0; *fmt != '\0'; fmt++)
  {
    if(*fmt == '%')
      switch(*++fmt)
      {
      case 'd':
	buf[dtoa(va_arg(arg_list, int), buf)] = '\0';
	cga_puts(buf);
	break;
      case 'u':
	buf[utoa(va_arg(arg_list, unsigned int), buf)] = '\0';
	cga_puts(buf);
	break;
      case 'x':
	buf[xtoa(va_arg(arg_list, unsigned int), buf)] = '\0';
	cga_puts(buf);
	break;
      case 'p':
	buf[ptoa(va_arg(arg_list, void *), buf)] = '\0';
	cga_puts(buf);
	break;
      case 's':
	cga_puts(va_arg(arg_list, char *));
	break;
      case 'c': cga_putc(va_arg(arg_list, char));
	break;
      }
    else
      cga_putc(*fmt);
  }

  va_end(arg);
}

/*======================================================================*/
/* ksprintf                                                             */
/*									*/
/* Desc: prints a formated string into the buffer pointed by "str".	*/
/*	 Valid formats are %c, %s, %d, %x and %p.			*/
/*									*/
/* Parm: str -> a pointer to the destination string (buffer)		*/
/* 	 fmt -> the format string					*/
/*	 ... -> arguments to be printed according to the format string	*/
/*======================================================================*/
void ksprintf(char *str, const char *fmt, ...)
{
  int i, len;
  char *arg;
  va_list arg_list;
 
  va_start(arg_list, fmt);

  for(i = 0; *fmt != '\0'; fmt++)
  {
    if(*fmt == '%')
      switch(*++fmt)
      {
      case 'd':
	i+= dtoa(va_arg(arg_list, int), &str[i]);
	break;
      case 'u':
	i+= utoa(va_arg(arg_list, unsigned), &str[i]);
	break;
      case 'x':
	i+= xtoa(va_arg(arg_list, unsigned), &str[i]);
	break;
      case 'p':
	i+= ptoa(va_arg(arg_list, void *), &str[i]);
	break;
      case 's':
	arg = va_arg(arg_list, char *);
	for(len = 0; arg[len] != '\0'; i++, len++)
	  str[i] = arg[len];
	break;
      case 'c': str[i++] = va_arg(arg_list, char);
	break;
      }
    else
      str[i++] = *fmt;
  }

  str[i] = '\0';

  va_end (arg);
}

/*======================================================================*/
/* dtoa		                                                        */
/*									*/
/* Desc: generates a string with a decimal representation of an integer.*/
/*									*/
/* Parm: d -> the integer						*/
/* 	 str -> a pointer to a buffer where the string will be stored	*/
/*									*/
/* Rtrn: the number of characters in the destination string		*/
/*======================================================================*/
int dtoa(int d, char *str)
{
  int i, j;

  if(d == 0)
  {
    str[0] = '0';
    return 1;
  }

  i = 0;

  if(d < 0)
  {
    d = -d;
    i++;
    str[0] = '-';
  }

  for(j = d; j != 0; i++, j /= 10);

  for(j = 0; d != 0; j++, d /= 10)
    str[i - 1 - j] = dec_digits[d % 10];
  
  return i;
}

/*======================================================================*/
/* utoa		                                                        */
/*									*/
/* Desc: generates a string with an unsigned decimal representation of	*/
/*	 an integer.							*/
/*									*/
/* Parm: d -> the integer						*/
/* 	 str -> a pointer to a buffer where the string will be stored	*/
/*									*/
/* Rtrn: the number of characters in the destination string		*/
/*======================================================================*/
int utoa(unsigned int u, char *str)
{
  unsigned int i, j;

  if(u == 0)
  {
    str[0] = '0';
    return 1;
  }

  i = 0;

  for(j = u; j != 0; i++, j /= 10);

  for(j = 0; u != 0; j++, u /= 10)
    str[i - 1 - j] = dec_digits[u % 10];
  
  return i;
}

/*======================================================================*/
/* xtoa		                                                        */
/*									*/
/* Desc: generates a string with a hexadecimal representation of an	*/
/*	 integer.							*/
/*									*/
/* Parm: x -> the integer						*/
/* 	 str -> a pointer to a buffer where the string will be stored	*/
/*									*/
/* Rtrn: the number of characters in the destination string		*/
/*======================================================================*/
int xtoa(unsigned int x, char *str)
{
  unsigned int i, j;

  str[0] = '0';
  str[1] = 'x';

  if(x == 0)
    return 1;

  for(i = 2, j = x; j != 0; i++, j >>= 4);

  for(j = 0; x != 0; j++, x >>= 4)
    str[i - 1 - j] = hex_digits[x & 0xf];

  return i;
}

/*======================================================================*/
/* ptoa		                                                        */
/*									*/
/* Desc: generates a string with a hexadecimal representation of a	*/
/*	 pointer.							*/
/*									*/
/* Parm: p -> the pointer						*/
/* 	 str -> a pointer to a buffer where the string will be stored   */
/*									*/
/* Rtrn: the number of characters in the destination string		*/
/*======================================================================*/
int ptoa(void *p, char *str)
{
  unsigned int i;

  str[0] = '0';
  str[1] = 'x';

  for(i = 0; i < sizeof(void *) * 2; i++, (int)p >>= 4)
    str[2 + sizeof(void *) * 2 - 1 - i] = hex_digits[(int)p & 0xf];

  return i + 2;
}
