/*======================================================================*/
/* INTERRUPT CONTROLLER (INTEL 8259) IMPLEMENTATION			*/
/*                                                                      */
/* Auth: Guto                                                           */
/*======================================================================*/
#include <pc/ix86.h>
#include <pc/pic.h>

/*======================================================================*/
/* pic_enable								*/
/*                                                                      */
/* Desc: Enables hardware interrupts.				 	*/
/*                                                                      */
/* Parm: mask -> a bit mask indicating the IRQ lines to be enabled	*/
/*		(1 = enable; 0 = don't care)			     	*/
/*======================================================================*/
void pic_enable(unsigned short mask)
{
  char master_mask;
  char slave_mask;

  master_mask = (char)mask;
  slave_mask  = (char)(mask >> 8);

  ix86_outb(PIC_MST_IMR, ix86_inb(PIC_MST_IMR) & ~master_mask);
  ix86_outb(PIC_SLV_IMR, ix86_inb(PIC_SLV_IMR) & ~slave_mask);
}

/*======================================================================*/
/* pic_disable								*/
/*                                                                      */
/* Desc: Disables hardware interrupts.			   		*/
/*                                                                      */
/* Parm: mask -> a bit mask indicating the interrupts to be disabled	*/
/*		(1 = disable; 0 = don't change)			    	*/
/*======================================================================*/
void pic_disable(unsigned short mask)
{
  char master_mask;
  char slave_mask;

  master_mask = (char)mask;
  slave_mask  = (char)(mask >> 8);

  ix86_outb(PIC_MST_IMR, ix86_inb(PIC_MST_IMR) | master_mask);
  ix86_outb(PIC_SLV_IMR, ix86_inb(PIC_SLV_IMR) | slave_mask);
}

/*======================================================================*/
/* pic_get_irr								*/
/*                                                                      */
/* Desc: Returns the contents of the IRR register (which interrupts are */
/*       pending).							*/
/*                                                                      */
/* Rtrn: the contents of the IRR register				*/
/*======================================================================*/
unsigned short pic_get_irr()
{
  unsigned short irr;

  /* Select IRR */
  ix86_outb(PIC_MST_CMD, PIC_SEL_IRR);
  ix86_outb(PIC_SLV_CMD, PIC_SEL_IRR);

  /* Get IRR */
  irr = ix86_inb(PIC_SLV_CMD) * 256 + ix86_inb(PIC_MST_CMD);

  return irr;
}

/*======================================================================*/
/* pic_get_isr								*/
/*                                                                      */
/* Desc: Returns the contents of the ISR register (which interrupts are */
/*       being serviced).						*/
/*                                                                      */
/* Rtrn: the contents of the ISR register				*/
/*======================================================================*/
unsigned short pic_get_isr()
{
  unsigned short isr;

  /* Select ISR */
  ix86_outb(PIC_MST_CMD, PIC_SEL_ISR);
  ix86_outb(PIC_SLV_CMD, PIC_SEL_ISR);

  /* Get ISR */
  isr = ix86_inb(PIC_SLV_CMD) * 256 + ix86_inb(PIC_MST_CMD);

  return isr;
}

/*======================================================================*/
/* pic_get_imr							        */
/*                                                                      */
/* Desc: Returns the contents of the IMR register (which interrupts are */
/*       enabled or disabled [0 means enabled, 1 means disabled]).	*/
/*                                                                      */
/* Rtrn: the contents of the IMR register				*/
/*======================================================================*/
unsigned short pic_get_imr()
{
  unsigned short imr;

  imr = ix86_inb(PIC_SLV_IMR) * 256 + ix86_inb(PIC_MST_IMR);

  return imr;
}

/*======================================================================*/
/* pic_set_base								*/
/*                                                                      */
/* Desc: Defines the ix86 interrupted to be invoked when IRQ0 is set.	*/
/* 	 Subsequent IRQs are mapped into subsequent ix86 interrupts.	*/
/*                                                                      */
/* Parm: base -> index in IDT for the first hardware interrupt		*/
/*======================================================================*/
void pic_set_base(unsigned char base)
{
  /* Configure Master PIC */
  ix86_outb(PIC_MST_CMD, PIC_ICW1);
  ix86_outb(PIC_MST_IMR, base);     /* ICW2 is the base */
  ix86_outb(PIC_MST_IMR, 0x04);     /* ICW3 = IRQ2 cascaded */
  ix86_outb(PIC_MST_IMR, PIC_ICW4);

  /* Configure Master PIC */
  ix86_outb(PIC_SLV_CMD, PIC_ICW1);
  ix86_outb(PIC_SLV_IMR, base + 8); /* ICW2 is the base */
  ix86_outb(PIC_SLV_IMR, 0x02);     /* ICW3 = cascaded from IRQ1 */
  ix86_outb(PIC_SLV_IMR, PIC_ICW4);  
}

/*======================================================================*/
/* pic_eoi								*/
/*                                                                      */
/* Desc: Sends an EOI (End Of Interrupt) to the PIC.			*/
/*======================================================================*/
void pic_eoi()
{
  /* Send EOI to master */
  ix86_outb(PIC_MST_CMD, PIC_EOI);

  /* If servicing slave IRQ send EOI to slave */
  if((pic_get_isr() & 0x0004) != 0)
    ix86_outb(PIC_SLV_CMD, PIC_EOI);
}
