#include "lzw.h"

#undef EOF

unsigned HashTable::find(unsigned pred, unsigned char foll)
{
	unsigned local, offset;
	Entry *ep; // pointer to current entry

	local = hash_function(pred,foll); // initial hash
	for (;;)
	{
		ep = &table[local];
		if ( (ep->predecessor == pred) && (ep->follower == foll) )
			return local;

		if ( ep->next == 0 )
			return NOT_FND;

		local = ep->next;
	}
}

unsigned HashTable::hash(unsigned pred, unsigned char foll)
{
	unsigned local, tempnext;
	Entry *ep;
	local = hash_function(pred,foll);
	if ( !table[local].used )
		return local;
	else
	{
		// if collision has occured
		local = eolist(local);

		// search for free entry from local + 101
		tempnext = (local + 101) & 0x0FFF;
		ep = &table[tempnext]; // initialize pointer
		while ( ep->used )
		{
				++tempnext;
				if ( tempnext == SIZE )
				{
					tempnext = 0; // handle wrap to beginning of table
					ep = table;   // address of first element of table
				}
				else
					++ep; // point to next element in table
		}

		// put new tempnext into last element in collision list
		table[local].next = tempnext;

		return tempnext;
	}
}

/*
 * getcode and putcode 'gallop' through input and output - they either
 * output two bytes or one depending on the contents of the buffer.
 */

unsigned InStream::getcode()
{
	unsigned localbuf, returnval;

	if (inbuf == EMPTY) // On code boundary
	{
		localbuf = get();
		if ( localbuf == EOF ) // H L1 byte - on code boundary
			return EOF;

		localbuf &= 0xFF;
		inbuf = get();
		if ( inbuf == EOF ) // L0 Hnext
			return EOF; // The file should always end on code boundary

		inbuf &= 0xFF;

		returnval = ((localbuf << 4) & 0xFF0) + ((inbuf >> 4) & 0x00F);

		inbuf &= 0x000F;
	}
	else // buffer contains nibble H
	{
		localbuf = get();
		if ( localbuf == EOF )
			return EOF;

		localbuf &= 0xFF;

		returnval = localbuf + ((inbuf << 8) & 0xF00);
		inbuf = EMPTY;

	}
	return returnval;
}

void OutStream::putcode(unsigned code)
{
	int localbuf;
	if (outbuf == EMPTY)
	{
		put( (code >> 4) & 0xFF ); // H L1
		outbuf = code & 0x000F;  // L0
	}
	else
	{
		put( ( (outbuf << 4) & 0xFF0) + ( (code >> 8) & 0x00F) ); // L0prev H
		put( code & 0x00FF);        // L1 L0
		outbuf = EMPTY;
	}
}

int InStream::get()
{
	if (pos == buffer.size())
		return EOF;
	else
		return buffer[pos++] & 0xFF;
}

void OutStream::put(int c)
{
	outsector[outcurrent++] = ( (char) c);
	if (outcurrent == SECTSIZE)
	{
		outcurrent = 0;
		buffer.append(outsector, SECTSIZE);
	}
}

// flushout makes sure fractional output buffer gets written
void OutStream::flushout()
{
	/*
	 * if there's still a byte waiting to be packed,
	 * stuff it in the buffer
	 */
	if (outbuf != EMPTY)
		outsector[outcurrent++] = (outbuf << 4) & 0xFF0;

	buffer.append(outsector, outcurrent);
}


void LZWCompressor::compress(const std::string &input, std::string &output)
{
	InStream in(input);
	OutStream out;
	unsigned c, code, localcode;
	int code_count = HashTable::SIZE - 256;

	c = in.get();
	code = table.find( HashTable::NO_PRED, c);  //initial code for table

	while ( (c = in.get()) != Stream::EOF )
	{
		localcode = table.find(code,c);
		if ( localcode != HashTable::NOT_FND )
			code = localcode;
		else
		{
			/*
			* when the above clause comes false,
			* the last known code has been found
			*/

			out.putcode(code);  // only update table if table isn't full

			if ( code_count )
			{
				table.put(code,c);
				--code_count;
			}
			// start loop again with the char that didn't fit into last string
			code = table.find(HashTable::NO_PRED, c);
		}
	}
	out.putcode(code);  //once EOF reached, always one code left unsent
	out.flushout();     // make sure everything's written
	output = out.getBuffer();
}

void LZWCompressor::expand(const std::string &input, std::string &output)
{
	InStream in(input);
	OutStream out;
	unsigned c, code, oldcode, incode, finchar,lastchar;
	bool unknown = false;
	int code_count = HashTable::SIZE - 256;
	HashTable::Entry *ep;

	stack.clear();

	code = oldcode = in.getcode();
	c = table.getEntry(code).follower;  // first code always known
	out.put(c);
	finchar = c;

	while ( (code = incode = in.getcode()) != Stream::EOF )
	{
		ep = &table.getEntry(code);  // initialize pointer
		if ( !ep->used ) // if code isn't known
		{
			lastchar = finchar;
			code = oldcode;
			unknown = true;
			ep = &table.getEntry(code); // re-initialize pointer
		}

		while (ep->predecessor != HashTable::NO_PRED)
		{
			stack.push( ep->follower); // decode string backwards into stack
			code = ep->predecessor;
			ep = &table.getEntry(code);
		}

		finchar = ep->follower;
		/*
		 * above loop terminates, one way or another, with
		 * table[code].follower = first char in string
		 */

		out.put(finchar);

		// pop anything stacked during code parsing
		while ( ! stack.empty() )
		{
			out.put( stack.pop() );
		}
		if ( unknown ) // if code isn't known
		{
			out.put(finchar = lastchar); // the follower char of last
			unknown = false;
		}
		if ( code_count )
		{
			table.put(oldcode,finchar);
			--code_count;
		}
		oldcode = incode;
	}
	out.flushout(); //write out fractional buffer, if any
	output = out.getBuffer();
}
