#ifndef LZW_H
#define LZW_H

class HashTable
{
public:
	struct Entry
	{
		Entry() : used(false), next(0), predecessor(0), follower(0) {}

		bool used;
		unsigned next;         // hi bit is 'used' flag
		unsigned predecessor;  // 12 bit code
		unsigned char follower;
	};

	static const unsigned SIZE = 4096;
	static const unsigned NO_PRED = 0xFFFF;
	static const unsigned NOT_FND = 0xFFFF;

public:
	HashTable() { reset(); }

	Entry &getEntry(unsigned pos) {return table[pos];}

	unsigned find(unsigned pred, unsigned char foll);

	void put(unsigned pred, unsigned foll)
	{
		Entry &ep = table[ hash(pred,foll) ];
		ep.next = 0;

		ep.used = true;
		ep.predecessor = pred;
		ep.follower = foll;
	}

	void reset()
	{
		for (unsigned i = 0; i < SIZE; i++)
			table[i] = Entry();

		for (unsigned i = 0; i < 256; i++)
			put(NO_PRED,i);
	}

private:
	Entry table[SIZE];

	// returns the mid square of pred + foll
	unsigned hash_function(unsigned pred, unsigned char foll)
	{
		long temp, local;  // 32 bit receiving field for local^2
		local = (pred + foll) | 0x0800;
		temp = local * local;
		local = (temp >> 6) & 0x0FFF;
		return local;   // middle 12 bits of result
	}

	unsigned hash(unsigned pred, unsigned char foll);

	// return last element in a collision list
	unsigned eolist(unsigned index)
	{
		int temp;
		while ( 0 != (temp = table[index].next) )
			index = temp;

		return index;
	}
};

class Stack
{
public:
	static const unsigned SIZE = HashTable::SIZE;
	static const unsigned EMPTY = 0xFFFF;

	Stack() : sp(0) {}

	void clear() { sp = 0; }

	void push(char c)
	{
		data[sp] = c;
		sp++;
	}

	char pop()
	{
		if (sp > 0)
		{
			sp--;
			return data[sp];
		}
		else
			return EMPTY;
	}

	bool empty() { return sp == 0; }

private:
	char data[SIZE];
	int sp;
};

struct Stream
{
	static const unsigned EMPTY = 0xFFFF;
	static const unsigned EOF = static_cast<unsigned>(-1);
};

#include <string>

class InStream : private Stream
{
public:
	InStream(const std::string &buffer)
	 : pos(0), buffer(buffer), inbuf(EMPTY) {	}

	unsigned getcode();
	int get();

private:
	const std::string buffer;
	unsigned pos;
	unsigned inbuf;
};


class OutStream : private Stream
{
public:
	static const unsigned SECTSIZE = 128;

	OutStream()
	 : outcurrent(0), outbuf(EMPTY) { }

	void putcode(unsigned code);
	void put(int c);
	void flushout();
	const std::string &getBuffer() {return buffer;}

private:
	std::string buffer;

	char outsector[SECTSIZE];
	int outcurrent;

	unsigned outbuf;
};

class LZWCompressor
{
public:
	void compress(const std::string &input, std::string &output);
	void expand(const std::string &input, std::string &output);

private:
	HashTable table;
	Stack stack;
};

#endif
