/*======================================================================*/
/* SYSTEM INITIALIZATION                                                */
/*                                                                      */
/* Desc: INIT is responsible for initializing the operating system. It  */
/*	 can relly on a basic node configuration carried out by SETUP.  */
/*       What INIT will actually do depends on the OS configuration for */
/*	 the specific application, but two things are mandatory:	*/
/*	 1 - Create the first process (TSS 0);				*/
/*	 2 - Preserve SETUP integrity.					*/
/*                                                                      */
/* Auth: Guto								*/
/*======================================================================*/
#include <elf.h>
#include <system/info.h>
#include <system/memory_map.h>
#include <system/kprintf.h>
#include <system/debug.h>
#include <system/setup.h>
#include <system/initializer.h>

__USING_SYS;

/*======================================================================*/
/* C PROTOTYPES                                                         */
/*======================================================================*/
extern "C"
{
    int main(char *boot_img);
    int _start(char *boot_img);
}

/*======================================================================*/
/* PROTOTYPES                                                         	*/
/*======================================================================*/
int init_loader(char *elf);

/*======================================================================*/
/* _start	          		                                */
/*                                                                      */
/* Desc: "_start" is INIT's entry point, so, if your compiler doesn't	*/
/*	 assume "_start" to be the entry point (GCC does), you somehow	*/
/*	 have to arrange this.						*/
/*======================================================================*/
int _start(char *boot_image)
{
    return (main(boot_image));
}

/*======================================================================*/
/* main		                                                        */
/*                                                                      */
/* Desc: call configuration specific functions to initialize the OS.    */
/*                                                                      */
/* Parm: boot_image -> a pointer to the image loaded from disk by BOOT  */
/*======================================================================*/
int main(char *boot_image)
{
    System_Info *si;

    debug(DB_INFO, DB_INIT, "Boot image loaded at %p\n", boot_image);

    /* Get System_Info */
    si = (System_Info *)SYS_INFO_LOG_ADDR;

    /* Initialize EPOS abstractions */
    Initializer initializer(si);

    /* Creates the first process */
    kprintf("  Creating first process: ");
    if(init_loader(&boot_image[si->boot_info.loader_off]) < 0) {
	kprintf("failed!\n");
	for(;;);
    }

    /* Return to SETUP */
    return 0;
}


/*======================================================================*/
/* init_loader	                                                        */
/*                                                                      */
/*======================================================================*/
int init_loader(char *elf)
{
    char *entry, *addr;
    int i, n_seg;
    unsigned size;

    debug(DB_TRACE, DB_INIT, "init_loader(elf=%p)\n", elf);

    /* Check integrity and get information about the ELF object */
    if(elf_get_hdr(elf, &entry, &n_seg) < 0) {
	debug(DB_WARNING, DB_INIT, "ELF image is corrupted!\n");
	return -1;
    }

    debug(DB_INFO, DB_INIT, "ELF={entry=%p,n_segs=%d}\n", entry, n_seg);

    /* Check if the entry point in the ELF header matches the expected */
    if((entry != 0) && (entry != (char *)0x00100000)) {
	debug(DB_WARNING, DB_INIT, "Invalid entry point in ELF image!\n");
	return -1;
    }

    /* Load the ELF object considering the addresses in the headers */
    for(i = 0; i < n_seg; i++)
	if(elf_get_seg(elf, i, &addr, &size) == 0) {
	    debug(DB_INFO, DB_INIT, "ELF SEG[%d]={addr=%p,size=%d})\n",
		  i, addr, size);
	    if(addr >= (char *)PHY_MEM_LOG_ADDR)
		return -1;
	    elf_load_seg(elf, i, addr);
	}

    /* Report */
    kprintf("done!\n");

    return 0;
}


