/*======================================================================*/
/* SYSTEM INITIALIZATION                                                */
/*                                                                      */
/* Desc: INIT is responsible for initializing the operating system. It  */
/*	 can relly on a basic node configuration carried out by SETUP.  */
/*       What INIT will actually do depends on the OS configuration for */
/*	 the specific application, but two things are mandatory:	*/
/*	 1 - Create the first process (TSS 0);				*/
/*	 2 - Preserve SETUP integrity.					*/
/*                                                                      */
/* Auth: Guto								*/
/*======================================================================*/
#include <sys/sys_info.h>
#include <sys/mem_map.h>
#include <sys/kprintf.h>
#include <sys/debug.h>
#include <sys/setup.h>
#include <sys/init.h>

#include <network_adapter.h>

/*======================================================================*/
/* C PROTOTYPES                                                         */
/*======================================================================*/
extern "C"
{
  int main(char *boot_img);
  int _start(char *boot_img);
}

/*======================================================================*/
/* PROTOTYPES                                                         	*/
/*======================================================================*/
int init_memory(Physical_Memory_Map *pmm);
int init_task();
int init_thread(Ix86_Log_Addr entry, Ix86_Phy_Addr pd, Ix86_Log_Addr stack);
int init_sync();
int init_comm();
int init_int();
int init_except();
int init_loader(char *elf);

/*======================================================================*/
/* _start	          		                                */
/*                                                                      */
/* Desc: "_start" is INIT's entry point, so, if your compiler doesn't	*/
/*	 assume "_start" to be the entry point (GCC does), you somehow	*/
/*	 have to arrange this.						*/
/*======================================================================*/
int _start(char *boot_image)
{
  return (main(boot_image));
}

/*======================================================================*/
/* main		                                                        */
/*                                                                      */
/* Desc: call configuration specific functions to initialize the OS.    */
/*                                                                      */
/* Parm: boot_image -> a pointer to the image loaded from disk by BOOT  */
/*======================================================================*/
int main(char *boot_image)
{
  System_Info *si;
  Boot_Info *bi;
  Physical_Memory_Map *pmm;

  debug(DB_INFO, DB_INIT, "Boot image loaded at %p\n", boot_image);

  /* Get System_Info */
  si = (System_Info *)SYS_INFO_LOG_ADDR;
  bi = &si->boot_info;
  pmm = &si->phy_mem_map;

  /* Say hi! */
  kprintf("Initializing the system:\n");

  kprintf("  Starting memory management: ");
  if(init_memory(pmm) < 0)
  {
    kprintf("failed!\n");
    for(;;);
  }

  kprintf("  Starting task management: ");
  if(init_task() < 0)
  {
    kprintf("failed!\n");
    for(;;);
  }

  kprintf("  Starting thread management: ");
  if(init_thread(0x00000000, pmm->sys_pd, pmm->app_hi) < 0)
  {
    kprintf("failed!\n");
    for(;;);
  }

  kprintf("  Starting synchronization management: ");
  if(init_sync() < 0)
  {
    kprintf("failed!\n");
    for(;;);
  }

  kprintf("  Starting communication management: ");
  if(init_comm() < 0)
  {
    kprintf("failed!\n");
    for(;;);
  }

  kprintf("  Starting interrupt management: ");
  if(init_int() < 0)
  {
    kprintf("failed!\n");
    for(;;);
  }

  kprintf("  Starting exception management: ");
  if(init_except() < 0)
  {
    kprintf("failed!\n");
    for(;;);
  }

  kprintf("  Creating first process: ");
  if(init_loader(&boot_image[bi->loader_off]) < 0)
  {
    kprintf("failed!\n");
    for(;;);
  }

  /* Return to SETUP */
}

/*======================================================================*/
/* init_memory	                                                        */
/*                                                                      */
/*======================================================================*/
int init_memory(Physical_Memory_Map *pmm)
{
  int i;
  Ix86_PTE *pts, *sys_pd;

  debug(DB_TRACE, DB_INIT,
	"init_memory(pmm={idt=%p,gdt=%p,pt=%p,pd=%p,info=%p,thr=%p,pts=%p,cod=%p,dat=%p,stk=%p,apl=%p,aph=%p})\n",
	pmm->idt, pmm->gdt, pmm->sys_pt, pmm->sys_pd, pmm->sys_info,
	pmm->thread_tab, pmm->phy_mem_pts, pmm->sys_code, pmm->sys_data,
	pmm->sys_stack, pmm->app_lo, pmm->app_hi);
  
  /* Flush TLB to ensure we've got the right memory organization */
  ix86_flush_tlb();

  /* Report */
  kprintf("single address space, %d Kbytes allocated to application, done!\n",
	  (pmm->app_hi - pmm->app_lo) / 1024);

  return 0;
}

/*======================================================================*/
/* init_task	                                                        */
/*                                                                      */
/*======================================================================*/
int init_task()
{
  /* Report */
  kprintf("not needed for this configuration!\n");

  return 0;
}

/*======================================================================*/
/* init_thread	                                                        */
/*                                                                      */
/*======================================================================*/
int init_thread(Ix86_Log_Addr entry, Ix86_Phy_Addr pd, Ix86_Log_Addr stack)
{
  Ix86_TSS *tss;

  debug(DB_TRACE, DB_INIT,
	"init_thread(entry=%d,pd=%p,stack=%p)\n", entry, pd, stack);

  /* Report */
  kprintf("single thread, done!\n");

  return 0;
}

/*======================================================================*/
/* init_sync	                                                        */
/*                                                                      */
/*======================================================================*/
int init_sync()
{
  /* Report */
  kprintf("not needed for this configuration!\n");

  return 0;
}

/*======================================================================*/
/* init_comm	                                                        */
/*                                                                      */
/*======================================================================*/
int init_comm(int *node_id)
{
  if(Current_Network_Adapter::init(node_id) < 0)
    return -1;

  /* Report */
  kprintf("user level Myrinet, done!\n");

  return 0;
}

/*======================================================================*/
/* init_int	                                                        */
/*                                                                      */
/*======================================================================*/
int init_int()
{
  /* Report */
  kprintf("not needed for this configuration!\n");

  return 0;
}

/*======================================================================*/
/* init_except	                                                        */
/*                                                                      */
/*======================================================================*/
int init_except()
{
  /* Report */
  kprintf("not needed for this configuration!\n");

  return 0;
}

/*======================================================================*/
/* init_loader	                                                        */
/*                                                                      */
/*======================================================================*/
int init_loader(char *elf)
{
  char *entry, *addr;
  int i, n_seg, size;

  debug(DB_TRACE, DB_INIT, "init_loader(elf=%p)\n", elf);

  /* Check integrity and get information about the ELF object */
  if(elf_get_hdr(elf, &entry, &n_seg) < 0)
    return -1;

  debug(DB_INFO, DB_INIT, "ELF={entry=%p,n_segs=%d}\n", entry, n_seg);

  /* Check if the entry point in the ELF header matches the expected */
  if((entry != 0) && (entry != (char *)0x00100000))
    return -1;

  /* Load the ELF object considering the addresses in the headers */
  for(i = 0; i < n_seg; i++)
    if(elf_get_seg(elf, i, &addr, &size) == 0)
    {
      debug(DB_INFO, DB_INIT, "ELF SEG[%d]={addr=%p,size=%d})\n",
	    i, addr, size);
      if(addr >= (char *)PHY_MEM_LOG_ADDR)
	return -1;
      elf_load_seg(elf, i, addr);
    }

  /* Report */
  kprintf("done!\n");

  return 0;
}


