/*======================================================================*/
/* INIT_LOADER.C                                                        */
/*                                                                      */
/* Dsc: Creates the first process and passes it the remaining of the	*/
/*	boot image in an extra segment mapped at 12 Mbytes.		*/
/* Cre: 96/07/24	Upd: 98/03/29					*/
/* Aut: Guto								*/
/*======================================================================*/
#include <aboelha/types.h>
#include <crt.h>
#include <ix86.h>
#include <aboelha/mem.h>
#include <aboelha/proc.h>
#include <aboelha/capability.h>
#include <aboelha/interface.h>
#include <em.h>
#include "init.h"

/*======================================================================*/
/* References to Kernel structures                                      */
/*======================================================================*/
extern Segment_Descriptor	Segment_Table [];
extern Task_Descriptor		Task_Descriptor_Table [];
extern Thread_Descriptor	*Thread_Descriptor_Table;
extern Thread_Descriptor	*Running;

/*======================================================================*/
/* PROTOTYPES                                                           */
/*======================================================================*/
void create_loader_code_seg(Elf_Manager *elf, Word *code);
void create_loader_data_seg(Elf_Manager *elf, Word *data);
void create_loader_stack_seg(int nres, Resource *res, Word *stack);
void create_loader_work_seg(Dwrd base_address, Word *servsers);
 
/*======================================================================*/
/* loader_init                                                          */
/*                                                                      */
/* Dsc: Creates the first process and passes the remaining of the boot	*/
/*	image to it in an extra segment mapped at 12 Mbytes.		*/
/*      Capabilities to all available resources are passed through the  */
/*      stack.                                                          */
/*======================================================================*/
int init_loader(int nres, Resource *res, Word *task_id, Word *thread_id)
{
  Word code;
  Word data;
  Word stack;
  Word work;
  Word resources;
  Dwrd base;
  char *elf;
  Capability cap;
  Elf_Manager em;

  /* Logical address of loader's ELF file */
  elf = (char *)(LOADER_PHY_ADDR + PHYSICAL_MEMORY_LOG_ADDR);

  em_create(&em, elf);
  if(!em_is_valid(&em))
  {
    printstr("Error: loader executable is not ELF!\n");
    for(;;);
  }

  create_loader_code_seg(&em, &code);
  create_loader_data_seg(&em, &data);
  create_loader_stack_seg(2, res, &stack);
  create_loader_work_seg(SERVERS_PHY_ADDR + PHYSICAL_MEMORY_LOG_ADDR,
    &work);

  if(task_create(code, data, task_id) < 0)
  {
    printstr("\nError: can not create loader's task!\n");
    for(;;);
  }

  if(capability_create(TASK_SERVER, *task_id, &cap) < 0)
  {
    printstr("Error: can not create loader's task capability!");
    for (;;);
  }
  
  /*
  if(thread_create(*task_id, stack, 0, sizeof(int) + nres * sizeof(Resource),
    THREAD_RUNNING, 3, 3, 3, thread_id) < 0)
  */
  if(thread_create(*task_id, stack, 0, 2 * sizeof(Resource) + sizeof(int),
    THREAD_RUNNING, 3, 3, 3, thread_id) < 0)
  {
    printstr("\nError: can not create loader's thread!\n");
    for(;;);
  }

  if(capability_create(THREAD_SERVER, *thread_id, &cap) < 0)
  {
    printstr("Error: can not create loader's thread capability!");
    for (;;);
  }

  /* First process will access the work segment at base 12 M */
  base = 0x00C00000;
  if(address_space_attach(*task_id, work, &base, SEG_RW) < 0)
  {
    printstr("\nError: can not attach work segment to loader's address space at ");
    printhex(base);
    printstr("!\n");
    for(;;);
  }

/*
  pt = (Dwrd *) (Segment_Table[servers].pt + 0x80000000);
  seg = (Byte *) ((pt [0] & 0xfffff000) + 0x80000000);

  printchar('\n');
  printstr("O pt ta apontando para:");
  printhex((int)pt);
  printchar('\n');
  printstr("O seg ta apontando para:");
  printhex((int)seg);
  printchar('\n');
  printstr("Printing the first eight bytes in app\n");
  for(i=0;i<8;i++) {
     printhex(seg[i]);
     printchar('\n');
  }
  printchar('\n');
  printstr("Ref: ");
  printhex(Segment_Table[servers].refs);
  printchar('\n');

  em_create(&em,(seg+sizeof(Dwrd)));

  if (!em_is_valid(&em))
  {
      printstr ("A aplicacao nao e elf\n");
  }
  else
  {
      printstr ("A aplicacao e elf\n");
  }
*/

}

/*======================================================================*/
/* create_loader_code_seg                                               */
/*                                                                      */
/* Desc: Creates loader code segment, and writes loader code in it.     */
/* Parm: aout -> pointer to loader executable in a.out form             */
/*       code <- capability for this segment                            */
/*======================================================================*/
void create_loader_code_seg(Elf_Manager *em, Word *code)
{
  Byte *seg;
  Dwrd *pt;
  Capability cap;
  int segment;
  Dwrd    size;
  Dwrd    addr;
  int   error;

  segment = em_get_next_segment (em, PT_LOAD, -1);

  em_get_loading_data(em, segment, &size, &addr);

  if((error = segment_create(0, size, code)) < 0)
  {
    printstr("Error: can not create loader's code segment!");
    for (;;);
  }
  
  pt = (Dwrd *)(Segment_Table[*code].pt + PHYSICAL_MEMORY_LOG_ADDR);
  seg = (Byte *)((pt[0] & 0xfffff000) + PHYSICAL_MEMORY_LOG_ADDR);

  em_load_segment(em,segment,(char *)seg);

  if(capability_create(SEGMENT_SERVER, *code, &cap) < 0)
  {
    printstr("Error: can not create loader's code segment capability!");
    for (;;);
  }
}

/*======================================================================*/
/* create_loader_data_seg                                               */
/*                                                                      */
/* Desc: Creates loader data segment, and writes loader data in it.     */
/* Parm: aout  -> pointer to loader executable in a.out form            */
/*       data  <-  capability for this segment                          */
/*======================================================================*/
void create_loader_data_seg (Elf_Manager *em, Word *data)
{
  Byte *seg;
  Dwrd *pt;
  Capability cap;
  int  segment;
  Dwrd  size,addr;
  int error;

  segment = em_get_next_segment (em, PT_LOAD, -1);
  segment = em_get_next_segment (em, PT_LOAD, segment);

  em_get_loading_data (em, segment, &size, &addr);

  if ((error = segment_create (0x0, size, data)) < 0)
  {
    printstr("Error: can not create loader's data segment!");
    for (;;);
  }

  pt = (Dwrd *)(Segment_Table[*data].pt + PHYSICAL_MEMORY_LOG_ADDR);
  seg = (Byte *) ((pt [0] & 0xfffff000) + PHYSICAL_MEMORY_LOG_ADDR);

  em_load_segment(em,segment,(char *)seg);

  if(capability_create(SEGMENT_SERVER, *data, &cap) < 0)
  {
    printstr("Error: can not create loader's data segment capability!");
    for (;;);
  }
}

/*======================================================================*/
/* create_loader_stack_seg                                              */
/*                                                                      */
/* Desc: Creates loader stack segment, stretching it to 4 Kbytes.       */
/*                                                                      */
/* Parm: cap <-  capability for this segment                            */
/*                                                                      */
/* Rtrn: nothing                                                        */
/*                                                                      */
/* Creation date: 07/25/96                                              */
/* Last updated : 09/30/96            Authors: Aboelha Development Team */
/*======================================================================*/
void create_loader_stack_seg(int nres, Resource *res, Word *stack)
{
  int i;
  Byte *seg;
  Dwrd *pt;
  Capability cap;

  if(segment_create(0x3ff000, 0x1000, stack) < 0)
  {
    printstr("Error: can not create loader's stak segment!");
    for (;;);
  }
  
  /* Get the logical address of this segment */
  pt = (Dwrd *)(Segment_Table[*stack].pt + PHYSICAL_MEMORY_LOG_ADDR);
  seg = (Byte *)((pt[1023] & 0xfffff000) + PHYSICAL_MEMORY_LOG_ADDR);

  res[0].tag = 0x00010203;
  res[0].capability.server.host = 0x04050607;
  res[0].capability.server.mailbox = 0x0809;
  res[0].capability.object = 0x0A0B;
  res[0].capability.rights = 0x0C0D;
  res[0].capability.check = 0x0E0F10111213;

  for(i = 0; i < nres * sizeof(Resource); i++)
    seg[0x1000 - nres * sizeof(Resource) + i] = ((Byte *)res)[i];
  *((int *)&seg[0x01000 - nres * sizeof(Resource) - sizeof(int)]) = nres;

  /*
  for(i = 0; i < 64; i++)
    seg[0x1000 - 64 + i] = i;
  */

  if(capability_create(SEGMENT_SERVER, *stack, &cap) < 0)
  {
    printstr("Error: can not create loader's stack segment capability!");
    for (;;);
  }
}

/*======================================================================*/
/* create_loader_work_seg                                               */
/*                                                                      */
/* Desc: Creates an extra segment to loader, which will contain         */
/*       system servers `a.outs', so that loader can read it and create */
/*       servers processes.                                             */
/*                                                                      */
/* Parm: base_address  -> base address where the `a.outs' have been     */
/*                        loaded                                        */
/*       cap  <-  capability for this segment                           */
/*                                                                      */
/* Rtrn: nothing                                                        */
/*                                                                      */
/* Creation date: 07/25/96                                              */
/* Last updated : 09/30/96            Authors: Aboelha Development Team */
/*======================================================================*/
void create_loader_work_seg(Dwrd base_address, Word *work)
{
  int   i;
  Byte *seg;
  Byte *aouts;
  Dwrd *pt;
  Capability cap;

  if (segment_create(0x0, 0x40000, work) < 0) /* work a.outs occupy 256 K */
  {
    printstr("Error: can not create loader's work segment!");
    for (;;);
  }

  /* get logical address of this segment */
  pt = (Dwrd *) (Segment_Table[*work].pt + PHYSICAL_MEMORY_LOG_ADDR);
  seg = (Byte *) ((pt [0] & 0xfffff000) + PHYSICAL_MEMORY_LOG_ADDR);
  aouts = (Byte *) base_address;

  for (i = 0; i < 0x40000; i++)
    seg [i] = aouts [i];

  if(capability_create(SEGMENT_SERVER, *work, &cap) < 0)
  {
    printstr("Error: can not create loader's work segment capability!");
    for (;;);
  }
}
