/*======================================================================*/
/* ELF LOADER IMPLEMENTATION						*/
/*									*/
/* Auth: Guto								*/
/*======================================================================*/
#include <elf.h>

/*======================================================================*/
/* elf_get_hdr								*/
/*									*/
/* Desc: Checks if the ELF object is valid and returns its entry point	*/
/*	 and the number of segments it contains.			*/
/*									*/
/* Parm: elf   -> a pointer to the ELF object				*/
/*	 entry <- ELF object's entry point				*/
/*	 n_seg <- number of segments in the ELF object			*/
/*									*/
/* Rtrn: 0 on success (entry and n_segs are valid)			*/
/*	 -1 if the object is not valid (entry and n_seg are undefined)	*/
/*======================================================================*/
int elf_get_hdr(char *elf, char **entry, int *n_seg)
{
  Elf32_Ehdr *hdr;

  hdr = (Elf32_Ehdr *)elf;

  if((hdr->e_ident[EI_MAG0] != ELFMAG0) ||
     (hdr->e_ident[EI_MAG1] != ELFMAG1) ||
     (hdr->e_ident[EI_MAG2] != ELFMAG2) ||
     (hdr->e_ident[EI_MAG3] != ELFMAG3))
    return -1;

  *entry = (char *)hdr->e_entry;
  *n_seg = hdr->e_phnum;

  return 0;
}

/*======================================================================*/
/* elf_get_seg								*/
/*									*/
/* Desc: Checks if the ELF segment is loadable and returns its loading  */
/*	 address and size.						*/
/*									*/
/* Parm: elf  -> a pointer to the ELF object				*/
/*	 seg  -> index into ELF objetc's segment table			*/
/*	 addr <- ELF segment start address				*/
/*	 size <- ELF segment size					*/
/*									*/
/* Rtrn: 0 on success (addr and size are valid)				*/
/*	 -1 if the segment is not valid (addr and size are undefined)	*/
/*======================================================================*/
int elf_get_seg(char *elf, int seg, char **addr, unsigned int *size)
{
  Elf32_Ehdr *hdr;
  Elf32_Phdr *pht;
  Elf32_Phdr *phdr;
  unsigned long	align_mask;

  hdr = (Elf32_Ehdr *)elf;
  pht = (Elf32_Phdr *)(elf + hdr->e_phoff);

  if(seg >= hdr->e_phnum)
    return -1;

  phdr = &pht[seg];
  if(phdr->p_type != PT_LOAD)
    return -1;

  align_mask = ~(phdr->p_align - 1);
  *addr = (char *)(phdr->p_vaddr & align_mask);
  *size = ((phdr->p_offset % phdr->p_align) +
           phdr->p_memsz + phdr->p_align - 1) & align_mask;

  return 0;
}

/*======================================================================*/
/* elf_load_seg								*/
/*									*/
/* Desc: Loads an ELF segment into the memory location given by "dst".  */
/*									*/
/* Parm: elf -> a pointer to the ELF object				*/
/*	 seg -> index into ELF objetc's segment table			*/
/*	 dts -> the location where the segment is to be loaded in	*/
/*									*/
/* Rtrn: 0 on success and -1 on failure					*/
/*======================================================================*/
int elf_load_seg(char *elf, int seg, char *dst)
{
  unsigned int i;
  char *src;
  Elf32_Ehdr *hdr;
  Elf32_Phdr *pht;
  Elf32_Phdr *phdr;

  hdr = (Elf32_Ehdr *)elf;
  pht = (Elf32_Phdr *)(elf + hdr->e_phoff);
  phdr = &pht[seg];

  src = &elf[phdr->p_offset];

  /* This have to be reviewd !!! */
  /* Align the destination pointer */
  /* dst+= (phdr->p_offset % phdr->p_align); */

  /* Initialized part of segment (text, rodata, data) */
  for(i = 0; i < phdr->p_filesz; i++)
    dst[i] = src[i];

  /* Zeroed part of segment (bss) */
  for (i = phdr->p_filesz; i < phdr->p_memsz; i++)
    dst[i] = 0;

  return 0;
}
