/*======================================================================*/
/* CGA VIDEO ADAPTER (CRT Controller Motorola 6845) 		        */
/*                                                                      */
/* Auth: Guto								*/
/*======================================================================*/
#include <pc/ix86.h>
#include <pc/cga.h>

/*======================================================================*/
/* GLOBALS	  	                                                */
/*======================================================================*/
/* Pointer to (CGA text) frame buffer. */
static unsigned short *FBuf = (unsigned short *)CGA_FBUF;

/*======================================================================*/
/* cga_set_fbuf  	                                                */
/*                                                                      */
/* Desc: define the CGA text frame buffer address.		        */
/*======================================================================*/
void cga_set_fbuf(unsigned long fbuf)
{
  FBuf = (unsigned short *)fbuf;
}

/*======================================================================*/
/* cga_clear								*/
/*                                                                      */
/* Desc: Clear the screen and restore cursor position to (0,0). 	*/
/*======================================================================*/
void cga_clear()
{
  int i;

  for (i = 0; i < CGA_LINES * CGA_COLUMNS; i++)
    FBuf[i] = 0x0020;

  ix86_outb(CGA_ADDR_REG, CGA_ADDR_CUR_POS_LO);
  ix86_outb(CGA_DATA_REG, 0);
  ix86_outb(CGA_ADDR_REG, CGA_ADDR_CUR_POS_HI);
  ix86_outb(CGA_DATA_REG, 0);
}

/*======================================================================*/
/* cga_gotoxy								*/
/*                                                                      */
/* Desc: Set the cursor position.                                	*/
/*                                                                      */
/* Parm: x  -> new x value                                              */
/*       y  -> new y value                                              */
/*======================================================================*/
void cga_gotoxy(int x, int y)
{
  int pos, old_x, old_y;

  if((x < 0) || (y < 0))
  {
    ix86_outb(CGA_ADDR_REG, CGA_ADDR_CUR_POS_LO);
    pos = ix86_inb(CGA_DATA_REG);
    ix86_outb(CGA_ADDR_REG, CGA_ADDR_CUR_POS_HI);
    pos |= ix86_inb(CGA_DATA_REG) << 8;
    old_x = pos % CGA_COLUMNS;
    old_y = pos / CGA_COLUMNS;
    if(x < 0)
      x = old_x;
    if(y < 0)
      y = old_y;
  }

  pos = y * CGA_COLUMNS + x;

  ix86_outb(CGA_ADDR_REG, CGA_ADDR_CUR_POS_LO);
  ix86_outb(CGA_DATA_REG, pos & 0xff);
  ix86_outb(CGA_ADDR_REG, CGA_ADDR_CUR_POS_HI);
  ix86_outb(CGA_DATA_REG, pos >> 8);
}

/*======================================================================*/
/* cga_putc		                                                */
/*                                                                      */
/* Desc: Prints a single character at the current cursor position.      */
/*                                                                      */
/* Parm: c  -> the character                                            */
/*======================================================================*/
void cga_putc(char c)
{
  int i, pos;

  /* Get cursor current position */
  ix86_outb(CGA_ADDR_REG, CGA_ADDR_CUR_POS_LO);
  pos = ix86_inb(CGA_DATA_REG);
  ix86_outb(CGA_ADDR_REG, CGA_ADDR_CUR_POS_HI);
  pos |= ix86_inb(CGA_DATA_REG) << 8;

  switch(c)
  {
  case '\n':
    pos = (pos + CGA_COLUMNS) / CGA_COLUMNS * CGA_COLUMNS;
  break;
  case '\t':
    pos = (pos + CGA_TAB_SIZE) / CGA_TAB_SIZE * CGA_TAB_SIZE;
  break;
  default:
    FBuf[pos++] = 0x0700 | c;
  }

  if(pos >= CGA_LINES * CGA_COLUMNS)
  {
    for(i = 0; i < (CGA_LINES - 1) * CGA_COLUMNS; i++)
      FBuf[i] = FBuf[i + CGA_COLUMNS];

    for(i = (CGA_LINES - 1) * CGA_COLUMNS; i < CGA_LINES * CGA_COLUMNS; i++)
      FBuf[i] = 0x0720;

    pos-= CGA_COLUMNS;
  }

  ix86_outb(CGA_ADDR_REG, CGA_ADDR_CUR_POS_LO);
  ix86_outb(CGA_DATA_REG, pos & 0xff);
  ix86_outb(CGA_ADDR_REG, CGA_ADDR_CUR_POS_HI);
  ix86_outb(CGA_DATA_REG, pos >> 8);
}

/*======================================================================*/
/* cga_puts		                                                */
/*                                                                      */
/* Desc: Prints a \0 terminated string, beggining at the current	*/
/*	 cursor position.						*/
/*                                                                      */
/* Parm: s -> a pointer to the string                                  	*/
/*======================================================================*/
void cga_puts(const char *s)
{
  while(*s != '\0')
    cga_putc(*s++);
}
