/*======================================================================*/
/* PC'S KEYBOARD CONTROLLER (Intel 8042) IMPLEMENTATION                 */
/*                                                                      */
/* Auth: Guto							     	*/
/*======================================================================*/
#include <pc/ix86.h>
#include <pc/keyb.h>

/*======================================================================*/
/* KEYB_SEND                                                            */
/*                                                                      */
/* Desc: Sends a byte to the keyboard controller according respecting	*/
/*	 the protocol.							*/
/*                                                                      */
/* Parm: data  -> the byte to be sent                                   */
/*                                                                      */
/* Rtrn: 0 on success, -1 on failure (timeout, no keyboard(!), etc.)    */
/*======================================================================*/
int keyb_send(unsigned char data)
{
  /* Wait till input buffer empty */
  while((ix86_inb(KEYB_STATUS) & KEYB_STAT_IN_BUF_FULL) != 0);

  ix86_outb(KEYB_DATA, data);

  /* Wait for keyboard controller reaction */
  while((ix86_inb(KEYB_STATUS) & KEYB_STAT_OUT_BUF_FULL) == 0);

  /* Did we got and ack? */
  if(ix86_inb(KEYB_DATA) == KEYB_ACK)
    return 0;
  else
    return -1;
}

/*======================================================================*/
/* KEYB_GET_SCAN_CODE 	                                                */
/*                                                                      */
/* Desc: Returns the scan code of the last pressed/released key.        */
/*                                                                      */
/* Rtrn: a key scan code                                                */
/*======================================================================*/
int keyb_get_scan()
{
  return(ix86_inb(KEYB_DATA));
}

/*======================================================================*/
/* KEYB_GET_STATUS                                                      */
/*                                                                      */
/* Desc: Returns the keyboard controller status.			*/
/*                                                                      */
/* Rtrn: the current keyboard controller status                         */
/*======================================================================*/
int keyb_get_status()
{
  return(ix86_inb(KEYB_STATUS));
}

/*======================================================================*/
/* KEYB_SET_LEDS                                                        */
/*                                                                      */
/* Desc: Set keyboard's LEDs state.                                     */
/*                                                                      */
/* Parm: leds_mask -> indicates which LEDs should be turned on          */
/*                                                                      */
/* Rtrn: 0 on success, -1 on failure (you should retry at least 3 times)*/
/*======================================================================*/
int keyb_set_leds(unsigned char leds_mask)
{
  if(keyb_send(KEYB_CMD_SET_LEDS) < 0)
    return -1;

  return(keyb_send(leds_mask));
}

/*======================================================================*/
/* KEYB_SET_RATE                                                        */
/*                                                                      */
/* Desc: Set keyboard repetition rate.                                  */
/*                                                                      */
/* Parm: delay  -> delay until first repetition (0-3)                   */
/*       rate   -> repetition rate (0-31)                               */
/*                                                                      */
/* Rtrn: 0 on success, -1 on failure (you should retry at least 3 times)*/
/*======================================================================*/
int keyb_set_rate(int delay, int rate)
{
  unsigned char rate_mask;

  rate_mask = (delay << 5) | rate;

  if(keyb_send(KEYB_CMD_SET_RATE) < 0)
     return -1;

  return(keyb_send(rate_mask));
}

/*======================================================================*/
/* KEYB_FLUSH                                                           */
/*                                                                      */
/* Desc: Flushes the keyboard controller's buffer.                      */
/*======================================================================*/
void keyb_flush()
{
  /* Until the input buffer is empty */
  do
  {
    /* Wait for controller ready */
    while((ix86_inb(KEYB_STATUS) & KEYB_STAT_OUT_BUF_FULL) != 0)
      /* Consume a character */
      ix86_inb(KEYB_DATA);
  } while ((ix86_inb(KEYB_STATUS) & KEYB_STAT_IN_BUF_FULL) != 0);
}

