/*======================================================================*/
/* NODE SETUP	          		                                */
/*                                                                      */
/* Desc: SETUP is responsible for bringing the node into a usable state.*/
/*       It setups several ix86 dependent data structures (IDT, GDT,	*/
/*	 TSSs, etc), a basic memory model, the FPU and, if configured	*/
/*	 to do so, the network. Note that setup deals with hardware	*/
/*	 dependent initializations. OS initialization is due to INIT.	*/
/*                                                                      */
/* Auth: Guto								*/
/*======================================================================*/
#include <elf.h>
#include <pc/ix86.h>
#include <pc/cga.h>
#include <pc/pic.h>
#include <pc/pci.h>
#include <system/kprintf.h>
#include <system/debug.h>
#include <system/bootp.h>
#include <system/setup.h>

/*======================================================================*/
/* PROTOTYPES                                                           */
/*======================================================================*/
void *memcpy(void *dst, const void *src, unsigned int n);
int  main(char *setup_addr, int setup_size, char *boot_image);
void setup_pic(unsigned int base);
void setup_gdt(Ix86_Phy_Addr addr, int n_threads);
void setup_idt(Ix86_Phy_Addr addr);
void setup_sys_pt(Physical_Memory_Map *pmm, int n_threads, int n_tasks,
		  int sys_code_size, int sys_data_size);
void setup_sys_pd(Physical_Memory_Map *pmm, unsigned int phy_mem_size,
		  Ix86_Phy_Addr pci_mem_phy_addr, unsigned int pci_mem_size);
void setup_sys_info(Ix86_Phy_Addr addr, System_Info *sys_info);
void setup_thread_tab(Ix86_Phy_Addr addr, int n_threads, Ix86_Phy_Addr pd,
		      Ix86_Log_Addr entry, Ix86_Log_Addr stack);
void setup_as_tab(Ix86_Phy_Addr addr, int n_tasks, Ix86_Phy_Addr pd);
int  get_bootp_info(unsigned short *node_id);
int  setup_myrinet(int unit);

void panic();
void page_fault();
void gpf();
void fpu();
void syscall();

/*======================================================================*/
/* _start	          		                                */
/*                                                                      */
/* Desc: In order to support larger boot images, the boot strap uses	*/
/*	 all memory below 640 Kb. In order to have more freedom to	*/
/* 	 setup the system, we will move SETUP to a more convinient	*/
/* 	 location. "_start" MUST be SETUP's entry point, so, if your	*/
/*	 compiler doesn't assume "_start" to be the entry point (GCC	*/
/*	 does), you somehow have to arrange this.			*/
/*	 The initial stack pointer is inherited from the boot strap	*/
/*	 (i.e., somewhere below 0x7c00). We can't "kprintf" here	*/
/*	 because the data segment is unreachable and "kprintf" has 	*/
/*	 static data. THIS FUNCTION MUST BE RELOCATABLE, because it	*/
/*	 won't run at the address it has been compiled for.		*/
/*======================================================================*/
void _start()
{
    char *boot_image, *elf, *entry, *addr, *dst;
    int i, n_seg;
    unsigned int size;
    Boot_Info *boot_info;

    /* Set EFLAGS */
    i = ix86_get_eflags();
    i &= EFLAGS_MASK;
    ix86_set_eflags(i);

    /* The boot strap loaded the boot image at BOOT_IMAGE_PHY_ADDR */
    boot_image = (char *)BOOT_IMAGE_PHY_ADDR;

    /* Get Boot_Info (System_Info is the first thing in the boot image) */
    boot_info = &((System_Info *)boot_image)->boot_info;

    /* BIOS gave us the extended memory size in K. We want it in bytes and */
    /* we SUPPOUSE we have the first Mb */
    /* boot_info->mem_size+= 1024; */
    boot_info->mem_size*= 1024;

    /* Check SETUP integrity and get information about its ELF structure */
    elf = &boot_image[boot_info->setup_off];
    if((elf_get_hdr(elf, &entry, &n_seg) < 0) || (n_seg != 1))
	panic();

    /* Test if we can access the address for which SETUP has been compiled */
    *entry = 'G';
    if(*entry != 'G')
	panic();

    /* Load SETUP considering the address in the ELF header if this won't */
    /* destroy the boot image */
    if((elf_get_seg(elf, 0, &addr, &size) < 0)
       || (addr <= &boot_image[boot_info->img_size]))
	panic();
    elf_load_seg(elf, 0, addr);

    /* Move the boot image to after SETUP, so there will be nothing else  */
    /* below SETUP to be preserved */
    dst = entry + size + sizeof(Ix86_Page); /* SETUP code + data + stack) */
    for(i = 0; i < boot_info->img_size; i++)
	dst[i] = boot_image[i];

    /* Setup a single page stack for SETUP after its data segment */
    /* (SP = "entry" + "size" + sizeof(Ix86_Page) */
    __asm__("movl %0, %%esp" : : "r" (entry + size + sizeof(Ix86_Page)));

    /* Pass the boot image to SETUP */
    __asm__("pushl %0" : : "r" (dst));

    /* Pass SETUP its size */
    __asm__("pushl %0" : : "r" (size));

    /* Pass SETUP its loading address */
    __asm__("pushl %0" : : "r" (addr));

    /* Call main() (the assembly is necessary because the compiler generates */
    /* relative calls and we need an absolute) */
    __asm__("call *%0" : : "r" (&main));
}

/*======================================================================*/
/* main		          		                                */
/*                                                                      */
/* Desc: We have to much to do here, you'd better to take a look at the	*/
/*	 remarks below. Important is the we've got a single Ix86_Page 	*/
/*	 for stack and we don't check its overflow, so be careful!	*/
/*                                                                      */
/* Parm: boot_image -> a pointer to the image loaded from disk by BOOT  */
/*======================================================================*/
int main(char *setup_addr, int setup_size, char *boot_image)
{
    int n_seg;
    unsigned int i, pci_mem_size, sys_code_size, sys_data_size, init_size;
    char *elf;
    char *sys_entry, *sys_code_addr, *sys_data_addr;
    char *init_entry, *init_addr;
    Ix86_Reg32 aux;
    Ix86_Phy_Addr pci_mem_phy_addr;
    System_Info *si;
    Boot_Info *bi;
    Physical_Memory_Map *pmm;

    /* Set the CGA text frame buffer so we can say hi */
    cga_set_fbuf(CGA_FBUF_PHY_ADDR);

    debug(DB_INFO, DB_SETUP, "Boot image loaded at %p\n", boot_image);

    /* Setup the interrupt controller */
    /* The BIOS sets hardware interrupts to 0x08-0x0f, but these IDT entries */
    /* are assigned to internal exceptions in the i(x>1)86 */
    /* We will remap interrupts to IDT_HARD_INT and then disable them */
    setup_pic(IDT_HARD_INT);

    /* Setup the PCI bus controller */
    if(pci_init((void **)&pci_mem_phy_addr, &pci_mem_size) < 0)
	kprintf("Warning: can't initialize PCI bus!\n");

    /* System_Info (actually Boot_Info) is the first thing in the boot image */
    si = (System_Info *)boot_image;
    bi = &si->boot_info;
    pmm = &si->phy_mem_map;

    /* If we didn't get our node's id from the boot image, we have to try to */
    /* get if from the eventual BOOPT reply used to boot up the system while */
    /* we didn't allocated (much) memory */
    if(bi->node_id == (unsigned short) -1)
	get_bootp_info(&bi->node_id);

    /* Check OS integrity and get the size of its code and data segments */
    elf = &boot_image[bi->system_off];
    if((elf_get_hdr(elf, &sys_entry, &n_seg) < 0) || (n_seg != 2)) {
	kprintf("Error: OS image is corrupted!\n");
	panic();
    }

    /* Get OS code segment size (aligned) */
    if((elf_get_seg(elf, 0, &sys_code_addr, &sys_code_size) < 0)
       || (sys_code_addr != (char *)SYS_CODE_LOG_ADDR)) {
	kprintf("Error: OS code segment is corrupted!\n");
	panic();
    }

    /* Get OS data segment size (aligned) */
    if((elf_get_seg(elf, 1, &sys_data_addr, &sys_data_size) < 0)
       || (sys_data_addr != (char *)SYS_DATA_LOG_ADDR)) {
	kprintf("Error: OS data segment is corrupted!\n");
	panic();
    }

    /* Check INIT integrity and get the its size (code+data) */
    elf = &boot_image[bi->init_off];
    if((elf_get_hdr(elf, &init_entry, &n_seg) < 0) || n_seg != 1) {
	kprintf("Error: INIT image is corrupted!\n");
	panic();
    }

    /* Get INIT code+data segment size (aligned) */
    if((elf_get_seg(elf, 0, &init_addr, &init_size) < 0)
       || (init_addr < setup_addr + setup_size)) {
	kprintf("Error: INIT code+data segment is corrupted!\n");
	panic();
    }

    /* Say hi! */
    kprintf("Setting up this node as follow: \n");
    kprintf("  Processor: 1 x Intel i%d86 (or compatible)\n", bi->cpu_type);
    kprintf("  Memory:    %d Kbytes\n", bi->mem_size/1024);
    kprintf("  Node Id:   ");
    if(bi->node_id != (unsigned) -1)
	kprintf("%x \n", bi->node_id);
    else
	kprintf("will get from the network!\n");
    kprintf("  Tasks:     %d (max)", bi->n_tasks);
    kprintf("  Threads:   %d (max)\n", bi->n_threads);
    kprintf("  Setup:     %d bytes", setup_size);
    kprintf("  Init:      %d bytes\n", init_size);
    kprintf("  OS code:   %d bytes", sys_code_size);
    kprintf("  OS data:   %d bytes\n", sys_data_size);

    /* Adjust the following sizes to pages */
    pci_mem_size = ix86_align_page(pci_mem_size) / sizeof(Ix86_Page);
    sys_code_size = ix86_align_page(sys_code_size) / sizeof(Ix86_Page);
    sys_data_size = ix86_align_page(sys_data_size) / sizeof(Ix86_Page);
    si->mem_size = ix86_align_page(bi->mem_size) / sizeof(Ix86_Page);
    si->mem_free = si->mem_size;

    /* Allocate (reserve) memory for all entities we have to setup. We will */
    /* start at the highest possible address to make possible a memory model */
    /* on which the application's logical and physical address spaces match. */
    /* IDT (1 Ix86_Page) */
    si->mem_free-= 1;
    pmm->idt = si->mem_free * sizeof(Ix86_Page);
 
    /* GDT (1 Ix86_Page) */
    si->mem_free-= 1;
    pmm->gdt = si->mem_free * sizeof(Ix86_Page);

    /* System Page Table (1 Ix86_Page) */
    si->mem_free-= 1;
    pmm->sys_pt = si->mem_free * sizeof(Ix86_Page);

    /* System Page Directory (1 Ix86_Page) */
    si->mem_free-= 1;
    pmm->sys_pd = si->mem_free * sizeof(Ix86_Page);

    /* System Info (1 Ix86_Page) */
    si->mem_free-= 1;
    pmm->sys_info = si->mem_free * sizeof(Ix86_Page);

    /* Thread (TSS) Table + dummy thread ((n_threads + 1) * Ix86_Page) */
    si->mem_free-= (bi->n_threads + 1);
    pmm->thread_tab = si->mem_free * sizeof(Ix86_Page);

    /* Address Space (Page Dir) Table (n_tasks * Ix86_Page) */
    si->mem_free-= bi->n_tasks;
    pmm->as_tab = si->mem_free * sizeof(Ix86_Page);

    /* Page tables to map the whole physical memory (NP/NPTE_PT * Ix86_Page) */
    /* NP = size of physical memory in pages */
    /* NPTE_PT = number of page table entries per page table */
    si->mem_free-= (si->mem_size + sizeof(Ix86_PT) / sizeof(Ix86_PTE) - 1)
	/ (sizeof(Ix86_PT) / sizeof(Ix86_PTE));
    pmm->phy_mem_pts = si->mem_free * sizeof(Ix86_Page);

    /* Page tables to map PCI address space (NP/NPTE_PT * Ix86_Page) */
    /* NP = size of PCI address space in pages */
    /* NPTE_PT = number of page table entries per page table */
    si->mem_free-= (pci_mem_size + sizeof(Ix86_PT) / sizeof(Ix86_PTE) - 1)
	/ (sizeof(Ix86_PT) / sizeof(Ix86_PTE));
    pmm->pci_mem_pts = si->mem_free * sizeof(Ix86_Page);

    /* OS code segment (in Ix86_Page) */
    si->mem_free-= sys_code_size;
    pmm->sys_code = si->mem_free * sizeof(Ix86_Page);

    /* OS data segment (in Ix86_Page) */
    si->mem_free-= sys_data_size;
    pmm->sys_data = si->mem_free * sizeof(Ix86_Page);

    /* OS stack segment (1 Ix86_Page) */
    si->mem_free-= 1;
    pmm->sys_stack = si->mem_free * sizeof(Ix86_Page);

    /* All memory bolow this is free to applications*/
    pmm->app_lo = 0;
    pmm->app_hi = si->mem_free * sizeof(Ix86_Page);

    /* Test if we didn't overlap SETUP and the boot image */
    if(si->mem_free * sizeof(Ix86_Page) <= (unsigned)setup_addr + setup_size) {
	kprintf("Error: SETUP would have been overwriten!\n");
	for(;;);
    }

    /* Zero the memory allocated to the system */
    for(i = si->mem_free * sizeof(Ix86_Page);
	i < si->mem_size * sizeof(Ix86_Page); i++)
	*((char *)i) = 0;

    /* Setup IDT */
    setup_idt(pmm->idt);

    /* Setup GDT */
    setup_gdt(pmm->gdt, bi->n_threads);

    /* Setup the System Page Table */
    setup_sys_pt(pmm, bi->n_threads, bi->n_tasks, sys_code_size, 
		 sys_data_size);

    /* Setup the System Page Directory and map physical memory */
    setup_sys_pd(pmm, si->mem_size, pci_mem_phy_addr, pci_mem_size);
    /* debug_page_table(&((Ix86_PTE *)pmm->sys_pd)[1024-128]); */

    /* Setup System Info */
    setup_sys_info(pmm->sys_info, si);

    /* Setup the TSS part of Thread Table */
    setup_thread_tab(pmm->thread_tab, bi->n_threads, pmm->as_tab,
		     0, pmm->app_hi);

    /* Setup the Address Space Table */
    setup_as_tab(pmm->as_tab, bi->n_tasks, pmm->sys_pd);

    /* Set IDTR (limit = 1 Ix86_Page) */
    ix86_set_idtr(sizeof(Ix86_Page) - 1, IDT_LOG_ADDR);

    /* Reload GDTR with its linear address (one more absurd from Intel!) */
    ix86_set_gdtr(sizeof(Ix86_Page) - 1, GDT_LOG_ADDR);

    /* Set CR3 (PDBR) register */
    ix86_set_cr3(pmm->sys_pd);

    /* Enable paging */
    aux = ix86_get_cr0();
    aux &= CR0_CLEAR_MASK; 
    aux |= CR0_SET_MASK;
    ix86_set_cr0(aux);

    debug(DB_INFO, DB_SETUP, "CR3=%x\n", ix86_get_cr3());

    /* The following  relative jump is to break the ix86 pre-fetch queue */
    /* (in case ix86_set_cr0() was a macro and didn't do it when returning) */
    /* and also to start using logical addresses */
    __asm__("this: jmp . + (next - this) + %0" : : "i" (PHY_MEM_LOG_ADDR)); 
    __asm__("next:"); 

    /* Reload segment registers with GDT_SYS_DATA */
    __asm__("" : : "a" (ix86_adjust_selector(GDT_SYS_DATA, 0)));
    __asm__("movw %ax, %ds");
    __asm__("movw %ax, %es");
    __asm__("movw %ax, %fs");
    __asm__("movw %ax, %gs");
    __asm__("movw %ax, %ss");

    /* Set stack pointer to its logical address */
    __asm__("orl %0, %%esp" : : "i" (PHY_MEM_LOG_ADDR));

    /* Adjust pointers that will still be used to their logical addresses */
    boot_image= (char *)(((unsigned long)(boot_image)) | PHY_MEM_LOG_ADDR);

    /* Flush TLB to ensure we've got the right memory organization */
    ix86_flush_tlb();

    /* Configure all devices we know about */
    kprintf("Configuring devices:\n");

    /* Nothing to do for CGA :-) */
    kprintf("  CGA video adapter: done!\n");

    /* Startup the FPU */
    kprintf("  FPU: ");
    ix86_init_fpu();
    kprintf("done!\n");

    /* Setup the first NIC (unit = 0) */
    kprintf("  Network: ");
/*     setup_myrinet(0); */
    kprintf("done!\n");
 
    /* Load OS code segment*/
    if(elf_load_seg(&boot_image[bi->system_off], 0, sys_code_addr) < 0) {
	kprintf("Error: OS code corrupted during SETUP!\n");
	panic();
    }

    /* Load OS data segment */
    if(elf_load_seg(&boot_image[bi->system_off], 1, sys_data_addr) < 0) {
	kprintf("Error: OS data corrupted during SETUP!\n");
	panic();
    }

    /* Load INIT */
    if(elf_load_seg(&boot_image[bi->init_off], 0, init_addr) < 0) {
	kprintf("Error: INIT corrupted during SETUP!\n");
	panic();
    }

    /* Give INIT a pointer to the boot image */
    __asm__("pushl %0" : : "r" (boot_image));

    /* Call INIT to initialize the OS */
    /* The minimum we expect from INIT is the creation of the first process */
    /* and the preservation of SETUP's code, data and stack */ 
    __asm__("call *%%ebx" : : "b"(init_entry));
/*  __asm__("call *%%ebx" : : "b"(init_entry) : "ebx"); */

    /* We are ready, from now on it's up to EPOS! */
    kprintf("Starting first process ...\n");

    /* Load TR register with the dummy TSS */
    ix86_set_tr(ix86_adjust_selector(GDT_TSS + bi->n_threads, 0));

    /* Switch to thread 0 (TSS 0) */
    __asm__("jmp 0");
    ix86_switch_tss(ix86_adjust_selector(GDT_TSS, 0));

    /* SETUP is now part of the free memory and this point should never be */
    /* reached, but, just for ... :-) */
    panic();

    /* Just to avoid the warning */
    return -1;
}

/*======================================================================*/
/* setup_pic			                                        */
/*                                                                      */
/* Desc: Remap and disable hardware interrupts.				*/
/*									*/
/* Parm: base -> offset in IDT for hardware interrupts			*/
/*======================================================================*/
void setup_pic(unsigned int base)
{
  debug(DB_TRACE, DB_SETUP, "setup_pic(base=%d)\n",base);

  /* Map IRQ0 into "base" */
  pic_set_base(base);

  /* Disable all interrupts */
  pic_disable(IRQ_ALL);
}

/*======================================================================*/
/* setup_idt	                                                        */
/*                                                                      */
/* Desc: Setup the IDT with panic for all entries but GPF and		*/
/*	 page-fault.						        */
/*                                                                      */
/* Parm: addr -> IDT PHYSICAL address 		                        */
/*======================================================================*/
void setup_idt(Ix86_Phy_Addr addr)
{
  int i;
  Ix86_IDT_Desc *idt;

  debug(DB_TRACE, DB_SETUP, "setup_idt(idt=%p)\n", addr);

  idt = (Ix86_IDT_Desc *)addr;

  /* Map all handlers to panic()  */
  for (i = 0; i < IX86_IDT_SIZE; i++)
    ix86_set_idt_entry(idt, i, GDT_SYS_CODE,
		       (Ix86_Reg32)&panic | PHY_MEM_LOG_ADDR,
		       SEG_FLAGS_INT | IX86_SEG_DPL1 | IX86_SEG_DPL2);

  /* Catch GPF, PAGE_FAULT and FPU */
  ix86_set_idt_entry(idt, IX86_EXC_GPF, GDT_SYS_CODE,
		     (Ix86_Reg32)&gpf | PHY_MEM_LOG_ADDR,
		     SEG_FLAGS_INT | IX86_SEG_DPL1 | IX86_SEG_DPL2);
  ix86_set_idt_entry(idt, IX86_EXC_PF,  GDT_SYS_CODE,
		     (Ix86_Reg32)&page_fault | PHY_MEM_LOG_ADDR,
		     SEG_FLAGS_INT | IX86_SEG_DPL1 | IX86_SEG_DPL2);
  ix86_set_idt_entry(idt, IX86_EXC_NODEV, GDT_SYS_CODE,
		     (Ix86_Reg32)&fpu | PHY_MEM_LOG_ADDR,
		     SEG_FLAGS_INT | IX86_SEG_DPL1 | IX86_SEG_DPL2);

  /* Catch system calls generated by accident */
  ix86_set_idt_entry(idt, 0x80, GDT_SYS_CODE,
		     (Ix86_Reg32)&syscall | PHY_MEM_LOG_ADDR,
		     SEG_FLAGS_INT | IX86_SEG_DPL1 | IX86_SEG_DPL2);

  debug(DB_INFO, DB_SETUP, "IDT[  %d]=%p%p (%p)\n", IX86_EXC_DIV0,
	*(int *)&idt[IX86_EXC_DIV0], *(((int *)&idt[IX86_EXC_DIV0]) + 1),
	panic);
  debug(DB_INFO, DB_SETUP, "IDT[%d]=%p%p (%p)\n", IX86_IDT_SIZE-1,
	*(int *)&idt[IX86_IDT_SIZE-1], *(((int *)&idt[IX86_IDT_SIZE-1]) + 1),
	panic);
}

/*======================================================================*/
/* setup_gdt		                                                */
/*                                                                      */
/* Desc: Setup the GDT, including entries for application and system	*/
/*	 code, data and stack and for the TSSs.                         */
/*                                                                      */
/* Parm: addr       -> GDT PHYSICAL address                           	*/
/* 	 n_threads  -> number of entries in Thread_Tab                 	*/
/*======================================================================*/
void setup_gdt(Ix86_Phy_Addr addr, int n_threads)
{
  int i;
  Ix86_Reg32 tss_base;
  Ix86_GDT_Desc *gdt;

  debug(DB_TRACE, DB_SETUP, "setup_gdt(gdt=%p)\n", addr);

  gdt = (Ix86_GDT_Desc *)addr;

  /* ix86_set_gdt_entry(gdt, index, base, limit, {P,DPL,S,TYPE}) */
  ix86_set_gdt_entry(gdt, GDT_NULL,      0,       0, 0);
  ix86_set_gdt_entry(gdt, GDT_APP_CODE,  0, 0xfffff, SEG_FLAGS_APP_CODE);
  ix86_set_gdt_entry(gdt, GDT_APP_DATA,  0, 0xfffff, SEG_FLAGS_APP_DATA);
  ix86_set_gdt_entry(gdt, GDT_APP_STACK, 0, 0xfffff, SEG_FLAGS_APP_DATA);
  ix86_set_gdt_entry(gdt, GDT_SYS_CODE,  0, 0xfffff, SEG_FLAGS_SYS_CODE);
  ix86_set_gdt_entry(gdt, GDT_SYS_DATA,  0, 0xfffff, SEG_FLAGS_SYS_DATA);
  ix86_set_gdt_entry(gdt, GDT_SYS_STACK, 0, 0xfffff, SEG_FLAGS_SYS_DATA);

  /* Set TSS's entries (n_threads + 1 for the dummy) */
  for (i = 0, tss_base = THREAD_TAB_LOG_ADDR;
       i < n_threads + 1;
       i++, tss_base += sizeof(Ix86_Page))
    ix86_set_gdt_entry(gdt, GDT_TSS + i, tss_base, 0xfff, SEG_FLAGS_TSS);

  debug(DB_INFO, DB_SETUP, "GDT[ %d]=%p%p\n", GDT_NULL,
	*(int *)&gdt[GDT_NULL], *(((int *)&gdt[GDT_NULL]) + 1));
  debug(DB_INFO, DB_SETUP, "GDT[ %d]=%p%p\n", GDT_APP_CODE,
	*(int *)&gdt[GDT_APP_CODE], *(((int *)&gdt[GDT_APP_CODE]) + 1));
  debug(DB_INFO, DB_SETUP, "GDT[ %d]=%p%p\n", GDT_APP_DATA,
	*(int *)&gdt[GDT_APP_DATA], *(((int *)&gdt[GDT_APP_DATA]) + 1));
  debug(DB_INFO, DB_SETUP, "GDT[ %d]=%p%p\n", GDT_APP_STACK,
	*(int *)&gdt[GDT_APP_STACK], *(((int *)&gdt[GDT_APP_STACK]) + 1));
  debug(DB_INFO, DB_SETUP, "GDT[ %d]=%p%p\n", GDT_SYS_CODE,
	*(int *)&gdt[GDT_SYS_CODE], *(((int *)&gdt[GDT_SYS_CODE]) + 1));
  debug(DB_INFO, DB_SETUP, "GDT[ %d]=%p%p\n", GDT_SYS_DATA,
	*(int *)&gdt[GDT_SYS_DATA], *(((int *)&gdt[GDT_SYS_DATA]) + 1));
  debug(DB_INFO, DB_SETUP, "GDT[ %d]=%p%p\n", GDT_SYS_STACK,
	*(int *)&gdt[GDT_SYS_STACK], *(((int *)&gdt[GDT_SYS_STACK]) + 1));
  debug(DB_INFO, DB_SETUP, "GDT[%d]=%p%p\n", GDT_TSS,
	*(int *)&gdt[GDT_TSS], *(((int *)&gdt[GDT_TSS]) + 1));
  debug(DB_INFO, DB_SETUP, "GDT[%d]=%p%p\n", GDT_TSS + n_threads,
	*(int *)&gdt[GDT_TSS + n_threads],
        *(((int *)&gdt[GDT_TSS + n_threads]) + 1));
}

/*======================================================================*/
/* setup_sys_pt		                                                */
/*                                                                      */
/* Desc: Setup the System Page Table	                                */
/*                                                                      */
/* Parm: pmm           -> physical memory map				*/
/*	 n_threads     -> number of entries in Thread_Tab		*/
/*	 n_tasks       -> number of entries in AS_Tab and in Task_Tab	*/
/*	 sys_code_size -> system code size in Ix86_Pages		*/
/*	 sys_data_size -> system data size in Ix86_Pages		*/
/*======================================================================*/
void setup_sys_pt(Physical_Memory_Map *pmm, int n_threads, int n_tasks,
		  int sys_code_size, int sys_data_size)
{
  unsigned int i;
  Ix86_PTE aux;
  Ix86_PTE *sys_pt;

  debug(DB_TRACE, DB_SETUP,
	"setup_sys_pt(pmm={idt=%p,gdt=%p,pt=%p,pd=%p,info=%p,thr=%p,as=%p,mem=%p,pci=%p,cod=%p,dat=%p,stk=%p,apl=%p,aph=%p},n_thr=%d,code_size=%d,data_size=%d)\n",
	pmm->idt, pmm->gdt, pmm->sys_pt, pmm->sys_pd, pmm->sys_info,
	pmm->thread_tab, pmm->as_tab, pmm->phy_mem_pts, pmm->pci_mem_pts,
	pmm->sys_code, pmm->sys_data, pmm->sys_stack, pmm->app_lo,
	pmm->app_hi, n_threads, sys_code_size, sys_data_size);

  /* Get the physical address for the System Page Table */
  sys_pt = (Ix86_PTE *)pmm->sys_pt;

  /* Clear the System Page Table */
  for(i = 0; i < sizeof(Ix86_PT)/sizeof(Ix86_PTE); i++)
    sys_pt[i] = 0;

  /* IDT */
  sys_pt[ix86_get_page(IDT_LOG_ADDR)] = pmm->idt | PAGE_FLAGS_SYS;

  /* GDT */
  sys_pt[ix86_get_page(GDT_LOG_ADDR)] = pmm->gdt | PAGE_FLAGS_SYS;

  /* Set an entry to this page table, so the system can access it later */
  sys_pt[ix86_get_page(SYS_PT_LOG_ADDR)] = pmm->sys_pt | PAGE_FLAGS_SYS;

  /* System Page Directory */
  sys_pt[ix86_get_page(SYS_PD_LOG_ADDR)] = pmm->sys_pd | PAGE_FLAGS_SYS;

  /* System Info */
  sys_pt[ix86_get_page(SYS_INFO_LOG_ADDR)] =
    pmm->sys_info | PAGE_FLAGS_SYS;

  /* Thread Table (each Thread includes a  TSS) */
  for(i = 0, aux = pmm->thread_tab;
      i < n_threads + 1;
      i++, aux += sizeof(Ix86_Page))
    sys_pt[ix86_get_page(THREAD_TAB_LOG_ADDR) + i] =
      aux | PAGE_FLAGS_SYS;

  /* Address Space Table (each AS is a page directory) */
  for(i = 0, aux = pmm->as_tab;
      i < n_tasks;
      i++, aux += sizeof(Ix86_Page))
    sys_pt[ix86_get_page(AS_TAB_LOG_ADDR) + i] =
      aux | PAGE_FLAGS_SYS;

  /* OS code */
  for(i = 0, aux = pmm->sys_code;
      i < sys_code_size;
      i++, aux += sizeof(Ix86_Page))
    sys_pt[ix86_get_page(SYS_CODE_LOG_ADDR) + i] =
      aux | PAGE_FLAGS_SYS;

  /* OS data */
  for(i = 0, aux = pmm->sys_data;
      i < sys_data_size;
      i++, aux += sizeof(Ix86_Page))
    sys_pt[ix86_get_page(SYS_DATA_LOG_ADDR) + i] =
      aux | PAGE_FLAGS_SYS;

  /* Set a single page for OS stack (who needs a stack?) */
  sys_pt[ix86_get_page(SYS_STACK_LOG_ADDR)] =
    pmm->sys_stack | PAGE_FLAGS_SYS;

  for(i = 0; i < sizeof(Ix86_PT)/sizeof(Ix86_PTE); i++)
      if(sys_pt[i])
	  debug(DB_INFO, DB_SETUP, "SPT[ %d]=%x\n", i, sys_pt[i]);
}

/*======================================================================*/
/* setup_sys_pd		                                                */
/*                                                                      */
/* Desc: Setups the System Page Directory and maps the physical memory	*/
/*	 starting at PHY_MEM_LOG_ADDR and also at "pmm->app_lo".	*/
/*                                                                      */
/* Parm: pmm	      -> physical memory map				*/
/* 	 phy_mem_size -> size of physical memory in Ix86_Pages          */
/* 	 pci_mem_size -> size of PCI address space			*/
/*======================================================================*/
void setup_sys_pd(Physical_Memory_Map *pmm, unsigned int phy_mem_size,
		  Ix86_Phy_Addr pci_mem_phy_addr, unsigned int pci_mem_size)
{
  int i, n_pts;
  Ix86_PTE *pts, *sys_pd;

  debug(DB_TRACE, DB_SETUP,
	"setup_sys_pd(pmm={idt=%p,...},mem_sz=%d,pci=%p,pci_sz=%d)\n",
	pmm->idt, phy_mem_size, pci_mem_phy_addr, pci_mem_size);

  /* Calculate the number of page tables needed to map the physical memory */ 
  n_pts = (phy_mem_size + (sizeof(Ix86_PT) / sizeof(Ix86_PTE)) - 1)
          / (sizeof(Ix86_PT) / sizeof(Ix86_PTE));
   
  /* Map all physical memory into the page tables pointed by phy_mem_pts */
  pts = (Ix86_PTE *)pmm->phy_mem_pts;
  for(i = 0; i < phy_mem_size; i++)
    pts[i] = (i * sizeof(Ix86_Page)) | PAGE_FLAGS_SYS;

  /* Setup the System Page Directory */
  sys_pd = (Ix86_PTE *)pmm->sys_pd;

  /* Attach all physical memory starting at PHY_MEM_LOG_ADDR */
  for(i = 0; i < n_pts; i++)
    sys_pd[ix86_get_dir(PHY_MEM_LOG_ADDR) + i] =
      (pmm->phy_mem_pts + i * sizeof(Ix86_Page)) | PAGE_FLAGS_SYS;

  /* Redefine page access control for the application memory */
  for(i = pmm->app_lo / sizeof(Ix86_Page);
      i < pmm->app_hi / sizeof(Ix86_Page);
      i++)
      pts[i] = (pts[i] & ~PAGE_FLAGS_SYS) | PAGE_FLAGS_APP;

  /* Attach application memory starting at "pmm->app_lo" */
  for(i = ix86_get_dir(ix86_align_dir(pmm->app_lo));
      i < ix86_get_dir(ix86_align_dir(pmm->app_hi));
      i++)
    sys_pd[i] = (pmm->phy_mem_pts + i * sizeof(Ix86_Page)) | PAGE_FLAGS_APP;

  /* Calculate the number of page tables needed to map the PCI memory */ 
  n_pts = (pci_mem_size + (sizeof(Ix86_PT) / sizeof(Ix86_PTE)) - 1)
          / (sizeof(Ix86_PT) / sizeof(Ix86_PTE));

  /* Map PCI addres space into the page tables pointed by pci_mem_pts */
  pts = (Ix86_PTE *)pmm->pci_mem_pts;
  for(i = 0; i < pci_mem_size; i++)
    pts[i] = (pci_mem_phy_addr + i * sizeof(Ix86_Page)) | PAGE_FLAGS_PCI;

  /* Attach PCI devices' memory to PCI_MEM_LOG_ADDR */
  for(i = 0; i < n_pts; i++)
    sys_pd[ix86_get_dir(PCI_MEM_LOG_ADDR) + i] =
      (pmm->pci_mem_pts + i * sizeof(Ix86_Page)) | PAGE_FLAGS_PCI;

  /* Map the system 4M logical address space at the top of the 4Gbytes */
  sys_pd[ix86_get_dir(SYS_CODE_LOG_ADDR)] = pmm->sys_pt | PAGE_FLAGS_SYS;

  for(i = 0; i < sizeof(Ix86_PT)/sizeof(Ix86_PTE); i++)
      if(sys_pd[i])
	  debug(DB_INFO, DB_SETUP, "PD[%d]=%x\n", i, sys_pd[i]);
}

/*======================================================================*/
/* setup_sys_info	                                                */
/*                                                                      */
/* Desc: Setups the system information block, which includes boot info	*/
/*	 and the physical memory map.					*/
/*                                                                      */
/* Parm: addr -> SYS_INFO PHYSICAL address				*/
/* 	 si   -> a pointer to the System_Info structure	 		*/
/*======================================================================*/
void setup_sys_info(Ix86_Phy_Addr addr, System_Info *si)
{
  System_Info *sys_info;
  
  debug(DB_TRACE, DB_SETUP,
	"setup_sys_info(info=%p,si={bi={mem=%d,...},pmm={idt=%p,...}})\n",
	addr, si->boot_info.mem_size, si->phy_mem_map.idt);

  sys_info = (System_Info *)addr;

  memcpy((char *)sys_info, (char *)si, sizeof(System_Info));
}

/*======================================================================*/
/* setup_thread_tab	                                                */
/*                                                                      */
/* Desc: Setups TSSs in the Thread Table. Each TSS is initialized 	*/
/*	 having in mind a single-task multi-thread model on which all	*/
/*	 threads run in SUPERVISOR mode with a 4Kb private stack	*/
/*	 allocated from the highest available address.			*/
/*                                                                      */
/* Parm: addr	   -> THREAD_TAB PHYSICAL address			*/
/*	 n_threads -> number of entries in Thread_Tab			*/
/* 	 pd        -> page directory PHYSICAL address			*/
/*		      (for thread's address space)			*/
/* 	 entry     -> thread's entry point (LOGICAL address)		*/
/* 	 stack     -> thread's stack (LOGICAL address)			*/
/*======================================================================*/
void setup_thread_tab(Ix86_Phy_Addr addr, int n_threads, Ix86_Phy_Addr pd,
		      Ix86_Log_Addr entry, Ix86_Log_Addr stack)
{
  int i;
  Ix86_TSS *tss;
  Ix86_Log_Addr tss_log;
  
  debug(DB_TRACE, DB_SETUP,
	"setup_thread_tab(thr=%p,n_thr=%d,pd=%p,ent=%p,stk=%p)\n",
	addr, n_threads, pd, entry, stack);

  for(i = 0,
	tss = (Ix86_TSS *)addr,
	tss_log = THREAD_TAB_LOG_ADDR;
      i < n_threads;
      i++,
	tss = (Ix86_TSS *)((int)tss + sizeof(Ix86_Page)),
	tss_log += sizeof(Ix86_Page),
	stack -= sizeof(Ix86_Page))
  {
    tss->back_link = 0x0000;
    tss->zero1     = 0x0000;
    /* System level stack shares the page with the TSS */
    tss->esp0      = tss_log + 0x1000; 
    tss->ss0       = ix86_adjust_selector(GDT_SYS_DATA, 0);
    tss->zero2     = 0x0000;
    tss->esp1      = 0x00000000;
    tss->ss1       = 0x0000;
    tss->zero3     = 0x0000;
    tss->esp2      = 0x00000000;
    tss->ss2       = 0x0000;
    tss->zero4     = 0x0000;
    tss->pdbr      = pd;
    tss->eip       = entry;
    tss->eflags    = 0x00003202;
    tss->eax       = 0x00000000;
    tss->ecx       = 0x00000000;
    tss->edx       = 0x00000000;
    tss->ebx       = 0x00000000;
    tss->esp       = stack;
    tss->ebp       = tss->esp;
    tss->esi       = 0x00000000;
    tss->edi       = 0x00000000;
    tss->es        = ix86_adjust_selector(GDT_APP_DATA, 0); /* 3 */
    tss->zero5     = 0x0000;
    tss->cs        = ix86_adjust_selector(GDT_APP_CODE, 0);
    tss->zero6     = 0x0000;
    tss->ss        = ix86_adjust_selector(GDT_APP_DATA, 0);
    tss->zero7     = 0x0000;
    tss->ds        = ix86_adjust_selector(GDT_APP_DATA, 0);
    tss->zero8     = 0x0000;
    tss->fs        = ix86_adjust_selector(GDT_APP_DATA, 0);
    tss->zero9     = 0x0000;
    tss->gs        = ix86_adjust_selector(GDT_APP_DATA, 0);
    tss->zero10    = 0x0000;
    tss->ldt       = 0x0000;
    tss->zero11    = 0x0000;
    tss->zero12    = 0x0000;
    tss->io_bmp    = 0xFFFF; /* Access to all IO ports */
  }    
}

/*======================================================================*/
/* setup_as_tab		                                                */
/*                                                                      */
/* Desc: Setups all address spaces (one per task) as copies of a	*/
/*	 template address space, usually the system address space.	*/
/*                                                                      */
/* Parm: addr	 -> AS_TAB PHYSICAL address				*/
/*	 n_tasks -> number of entries in Thread_Tab			*/
/* 	 pd	 -> template address space (a page directory PHYSICAL	*/
/*		    address)						*/
/*======================================================================*/
void setup_as_tab(Ix86_Phy_Addr addr, int n_tasks, Ix86_Phy_Addr pd)
{
  int i;
  Ix86_PD *as_tab;

  debug(DB_TRACE, DB_SETUP,
	"setup_as_tab(as=%p,n_tsk=%d,pd=%p)\n",
	addr, n_tasks, pd);

  as_tab = (Ix86_PD *)addr;

  /* Copy SYS_PD to each entry of AS_TAB */
  for(i = 0; i < n_tasks; i++)
    memcpy(&as_tab[i], (void *)pd, sizeof(Ix86_PD));
}

/*======================================================================*/
/* get_bootp_info                                                       */
/*                                                                      */
/* Desc: Scans the BOOTP reply used to boot up the system for "vendor"	*/
/*                                                                      */
/* Parm: node_id <- this node's logical id			        */
/*                                                                      */
/* Rtrn: 0 if tag found and node_id was set, -1 otherwise	        */
/*======================================================================*/
int get_bootp_info(unsigned short *node_id)
{
  int i;
  unsigned char *reply;

  debug(DB_TRACE, DB_SETUP, "get_bootp_info(nid=%p)\n", node_id);

  /* Scan BOOTP reply for RFC1084 Magic Cookie */
  for(i = 0, reply = (unsigned char *)BOOTP_REPLY_PHY_ADDR;
      i < BOOTP_REPLY_SIZE;
      i += sizeof(unsigned int))
    if(*(unsigned int *)&reply[i] == ix86_htonl(BOOTP_MAGIC))
      break;

  /* No BOOTP reply found */
  if(i == BOOTP_REPLY_SIZE)
    return -1;

  /* Scan BOOTP reply for our tag  */
  for(; i < BOOTP_REPLY_SIZE; i++)
    if((reply[i] == BOOTP_EPOS_TAG))
    {
      /* Tag value can be 8, 16, or 32 bits long */
      switch(reply[i+1])
      {
        case 1: *node_id = (int)reply[i+2];
	break;
        case 2: *node_id = ix86_htons(*(unsigned short *)&reply[i+2]);
	break;
        case 4: *node_id = ix86_htonl(*(unsigned long *)&reply[i+2]);
	break;
        default: return -1;
      }
      return 0;
    }

  /* No BOOTP_EPOS_TAG found */
  return -1;
}

/*======================================================================*/
/* panic                                                                */
/*                                                                      */
/* Desc: Function called if something fails during setup, including	*/
/*	 untreated interrupts.					        */
/*======================================================================*/
void panic()
{
  unsigned short *video;

  /* Set a pointer to GCA text frame buffer so we can say something */
  video = (unsigned short *) CGA_FBUF_PHY_ADDR;

  video [0] = 'P' | 0x1f00;
  video [1] = 'A' | 0x1f00;
  video [2] = 'N' | 0x1f00;
  video [3] = 'I' | 0x1f00;
  video [4] = 'C' | 0x1f00;
  video [5] = '!' | 0x1f00;

  for(;;);
}

/*======================================================================*/
/* page_fault                                                           */
/*                                                                      */
/* Desc: Page fault exception handler (what a handling! :-)).           */
/*                                                                      */
/* Parm: exception stack and error code pushed by the CPU		*/
/*======================================================================*/
void page_fault(Ix86_Reg32 eip, Ix86_Reg32 cs, Ix86_Reg32 eflags,
		Ix86_Reg32 esp3, Ix86_Reg32 ss, Ix86_Reg32 error)
{  
  debug(DB_ERROR, DB_SETUP,
	"this thread generated an invalid address and will be terminated!\n");
  debug(DB_ERROR, DB_SETUP, "thread=%d, address=%p\n",
	 (ix86_get_tr() >> 3) - GDT_TSS, ix86_get_cr2());
  debug(DB_ERROR, DB_SETUP,
	 "context={cs=%x,ip=%p,flg=%x,ss=%x,sp=%p,err=%x}\n",
	 cs, eip, eflags, ss, esp3, error);

  panic();
}

/*======================================================================*/
/* gpf                                                                  */
/*                                                                      */
/* Desc: GPF exception handler (what a handling! :-)).   	        */
/*                                                                      */
/* Parm: exception stack and error code pushed by the CPU		*/
/*======================================================================*/
void gpf(Ix86_Reg32 eip, Ix86_Reg32 cs, Ix86_Reg32 eflags,
	 Ix86_Reg32 esp3, Ix86_Reg32 ss, Ix86_Reg32 error)
{  
    short ds = 0;

    __asm__("movw %%ds, %0" : "=o" (ds) : );

    debug(DB_ERROR, DB_SETUP,
	  "this thread caused a GPF and will be terminated!\n");
    debug(DB_ERROR, DB_SETUP, "thread=%d ", (ix86_get_tr() >> 3) - GDT_TSS);
    debugn(DB_ERROR, DB_SETUP,
	   "context={cs=%x,ds=%x,ip=%p,flg=%x,ss=%x,sp=%p,err=%x}\n",
	   cs, ds, eip, eflags, ss, esp3, error);
    
    panic();
}

/*======================================================================*/
/* fpu                                                                  */
/*                                                                      */
/* Desc: FPU exception handler.				   	        */
/*======================================================================*/
void fpu()
{  
  debug(DB_TRACE, DB_SETUP, "FPU generated an interrupt!\n");

  __asm__("clts");
}

/*======================================================================*/
/* syscall                                                              */
/*                                                                      */
/* Desc: FPU exception handler.				   	        */
/*======================================================================*/
void syscall()
{  
  debug(DB_ABORT, DB_SETUP, "Error: system call invoked but no OS loaded!\n");

  panic();
}

void *memcpy(void *dst, const void *src, unsigned int n)
{
    int i;
    for(i = 0; i < n; i++)
	((char *)dst)[i] = ((char *)src)[i];
    return dst;
}
