#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <sys/stat.h>

#define MAX_LINE_SIZE  256
#define MAX_ADDR_SIZE 16
#define MAX_NUMBER_OF_NODES 256
#define MAX_ROUTE_SIZE 8

typedef unsigned long long Myrinet_Address;
typedef struct {
  Myrinet_Address from;
  Myrinet_Address to;
  char path[MAX_ROUTE_SIZE];
} Myrinet_Route;

int get_route(FILE *file, Myrinet_Route *route);
Myrinet_Address get_address(char *str);
int get_node_id(Myrinet_Address addr);

Myrinet_Address address_table[MAX_NUMBER_OF_NODES];
char routing_table[MAX_NUMBER_OF_NODES][MAX_NUMBER_OF_NODES][MAX_ROUTE_SIZE];
unsigned int N_NODES;
unsigned int ROUTE_LEN;

int main(int argc, char *argv[])
{
  FILE *r_file, *c_file;
  int verbose;
  unsigned int i, j, k, line_nr, err, from_node_id, to_node_id, n_nodes;
  Myrinet_Route route;

  /* Parse the command line */   
  if(argc < 5) {
    fprintf(stderr,
      "Usage: %s [-v] <#nodes> <route_len> <route_file> <c_file>\n", argv[0]);
    fprintf(stderr,
      "       %s [-v] <#nodes> <route_len> - - # stdin stdout\n", argv[0]);
    return -1;
  }

  if(argc == 6) {
    verbose = 1;
    argv++;
  }
  else
    verbose = 0;

  N_NODES = atoi(argv[1]);
  if(N_NODES == 0) {
    fprintf(stderr, "Error: number of nodes can't be \"%s\"!\n", argv[1]);
    return -1;
  }

  ROUTE_LEN = atoi(argv[2]);
  if(ROUTE_LEN == 0) {
    fprintf(stderr, "Error: route length can't be \"%s\"!\n", argv[2]);
    return -1;
  }

  if(!strcmp(argv[3], "-"))
    r_file = stdin;
  else
    if(!(r_file = fopen(argv[3], "r"))) {
      fprintf(stderr, "Error: can't open routes file \"%s\"!\n",
	      argv[3]);
      return -1;
    }

  if(!strcmp(argv[4], "-"))
    c_file = stdout;
  else
    if(!(c_file = fopen(argv[4], "w"))) {
      fprintf(stderr, "Error: can't write to file \"%s\"!\n", argv[4]);
      return -1;
    }

  /* Parse the routing table */   
  n_nodes = 0;
  for(line_nr = 1, err = get_route(r_file, &route); !err;
      line_nr++, err = get_route(r_file, &route)) {
    if(route.from != route.to) {
      from_node_id = get_node_id(route.from);
      if(from_node_id < 0) {
	fprintf(stderr, "Parse error at line %d: too many nodes!\n", line_nr);
	return -1;
      }
      
      if(!address_table[from_node_id])
	address_table[from_node_id] = route.from;
      
      to_node_id = get_node_id(route.to);
      if(to_node_id < 0) {
	fprintf(stderr, "Parse error at line %d: too many nodes!\n", line_nr);
	return -1;
      }

      if(from_node_id > n_nodes)
	n_nodes = from_node_id;
      
      if(!address_table[to_node_id])
	address_table[to_node_id] = route.to;
      
      if(routing_table[from_node_id][to_node_id][0]) {
	fprintf(stderr,
          "Parse error at line %d: duplicated route from [%Lx] to [%Lx]!\n",
	  line_nr, route.from, route.to);
	return -1;
      }
      memcpy(routing_table[from_node_id][to_node_id], route.path,
	     MAX_ROUTE_SIZE);

      if(verbose) {
	fprintf(stderr, "node%d[%Lx] -> node%d[%Lx] =  ",
		from_node_id, address_table[from_node_id],
		to_node_id, address_table[to_node_id]);
	for(i = 0; (i < MAX_ROUTE_SIZE) && route.path[i]; i++)
	  fprintf(stderr, "%d ", routing_table[from_node_id][to_node_id][i]);
	fprintf(stderr, "\n");
      }
    }
  }

  if(n_nodes == 0) {
    fprintf(stderr, "Error: empty or corrupted route file!\n");
    return -1;
  }
  
  n_nodes++;
  if(verbose)
    fprintf(stderr, "Total of %d nodes found\n", n_nodes);

  if(n_nodes != N_NODES) {
    fprintf(stderr,
	    "Warning: %d nodes found in the route file, %d specified!\n",
	    n_nodes, N_NODES);
  }

  /* Generate the output file */
  fprintf(c_file, "#define MYRINET_ADDRESS_TABLE {\\\n");
  for(i = 0; i < n_nodes; i++)
    fprintf(c_file, "  0x%LxLL,\\\n", address_table[i]);
  fprintf(c_file, "}\n\n");

  fprintf(c_file, "#define MYRINET_ROUTING_TABLE {\\\n");
  for(i = 0; i < n_nodes; i++) {
    fprintf(c_file, "  {\\\n");
    for(j = 0; j < n_nodes; j++) {
      fprintf(c_file, "    {");
      for(k = 0; k < ROUTE_LEN; k++) {
	fprintf(c_file, "%d", routing_table[i][j][k]);
	if(k < ROUTE_LEN - 1)
	  fprintf(c_file, ", ");
      }
      fprintf(c_file, "}");
      if(j < n_nodes - 1)
        fprintf(c_file, ",\\\n");
      else
        fprintf(c_file, "\\\n");
    }
    fprintf(c_file, "  }");
    if(i < n_nodes - 1)
      fprintf(c_file, ",\\\n");
  }
  fprintf(c_file, "\\\n};\\\n");

  fclose(r_file);
  fclose(c_file);

  return 0;
}

int get_route(FILE *file, Myrinet_Route *route) {
  char buf[MAX_LINE_SIZE];
  int i;
  char *from_name, *from_addr, *to_name, *to_addr, *path;

  if(!fgets(buf, MAX_LINE_SIZE, file))
    return -1;

  from_name = strtok(buf, " ");
  from_addr = strtok(NULL, " ");
  to_name = strtok(NULL, " ");
  to_addr = strtok(NULL, " ");
  path = strtok(NULL, " ");

  if(path == NULL)
    return -1;

  for(i = 0; i < MAX_ROUTE_SIZE; i++)
    if(path) {
      route->path[i] = (atoi(path) & 0x3f) | 0x80; 
      path = strtok(NULL, " ");
    } else
      route->path[i] = 0;

  route->from = get_address(from_addr);
  route->to = get_address(to_addr);

  return 0;
}

Myrinet_Address get_address(char *str)
{
  Myrinet_Address address;
  char buf1[MAX_LINE_SIZE], buf2[MAX_ADDR_SIZE], buf3[MAX_ADDR_SIZE];

  strncpy(buf1, str, MAX_LINE_SIZE);

  sprintf(buf2, "%s", strtok(buf1, ":"));
  sprintf(buf2 + 2, "%s", strtok(NULL, ":"));
  sprintf(buf3, "%s", strtok(NULL, ":"));
  sprintf(buf3 + 2, "%s", strtok(NULL, ":"));
  sprintf(buf3 + 4, "%s", strtok(NULL, ":"));
  sprintf(buf3 + 6, "%s", strtok(NULL, "\n"));
  buf2[4] = '\0';
  buf3[8] = '\0';

  address = (Myrinet_Address)strtoul(buf2, NULL, MAX_ADDR_SIZE);
  address = address << 32 |
    (Myrinet_Address)strtoul(buf3, NULL, MAX_ADDR_SIZE);

  return address;
}  

int get_node_id(Myrinet_Address addr)
{
  unsigned int i;
   
  for(i = 0; (i < N_NODES) && address_table[i]; i++)
    if(address_table[i] == addr)
      break;
  if(i == N_NODES)
    i = (unsigned) -1;

  return i;
}

