#include <stdio.h>
#include <stdlib.h>
#include <sys/stat.h>
#include <unistd.h>
#include <elf.h>

int main(int argc, char *argv[])
{
  unsigned int i,size;
  char *buffer;
  FILE *file;
  struct stat status;
  Elf32_Ehdr *ehdr;
  Elf32_Phdr *phdr;
  Elf32_Shdr *shdr;

  if (argc != 2) {
    fprintf(stderr, "Usage: %s <elf file>!\n", argv[0]);
    return 1;
  }

  file = fopen(argv[1], "r");
  if (file == NULL) {
    fprintf(stderr, "Error: can not open file %s!\n", argv[1]);
    return 1;
  }

  if(stat(argv[1], &status) < 0) {
    fprintf(stderr, "Error: can not stat file %s!\n", argv[1]);
    return 1;
  }
  size = status.st_size;

  buffer = (char *) malloc (size);

  if(fread(buffer, 1, size, file) != size) {
    fprintf(stderr, "Error: can not load file %s!\n", argv[1]);
    return 1;
  }

  ehdr = (Elf32_Ehdr *) buffer;
  if((ehdr->e_ident[EI_MAG0] != ELFMAG0) ||
     (ehdr->e_ident[EI_MAG1] != ELFMAG1) ||
     (ehdr->e_ident[EI_MAG2] != ELFMAG2) ||
     (ehdr->e_ident[EI_MAG3] != ELFMAG3)) {
    fprintf(stderr, "Error: %s is not an ELF32 file!\n", argv[1]);
    return 1;
  }

  printf("ELF32 Header =>\n");
  printf("  Magic number: %p\n", ehdr->e_ident);
  printf("  Object file type: 0x%x\n", ehdr->e_type);
  printf("  Architecture: 0x%x\n", ehdr->e_machine);
  printf("  Object file version: 0x%x\n", ehdr->e_version);
  printf("  Entry point virtual address: 0x%x\n", ehdr->e_entry);
  printf("  Program header table file offset: 0x%x\n", ehdr->e_phoff);
  printf("  Section header table file offset: 0x%x\n", ehdr->e_shoff);
  printf("  Processor-specific flags: 0x%x\n", ehdr->e_flags);
  printf("  ELF header size in bytes: %d\n", ehdr->e_ehsize);
  printf("  Program header table entry size: %d\n", ehdr->e_phentsize);
  printf("    Size of Elf32_Phdr: %d\n", sizeof(Elf32_Phdr));
  printf("  Program header table entry count: 0x%x\n", ehdr->e_phnum);
  printf("  Section header table entry size: %d\n", ehdr->e_shentsize);
  printf("    Size of Elf32_Shdr: %d\n", sizeof(Elf32_Shdr));
  printf("  Section header table entry count: 0x%x\n", ehdr->e_shnum);
  printf("  Section header string table index: 0x%x\n", ehdr->e_shstrndx);
  printf("\n");

  if(ehdr->e_type != ET_EXEC) {
    fprintf(stderr, "Warning: %s is not an ELF32 executable!\n", argv[1]);
  }

  if(ehdr->e_machine != EM_386) {
    fprintf(stderr,
      "Warning: %s is not an ELF32 executable for this architecture[ix86]!\n",
      argv[1]);
  }

  for(i = 0; i < ehdr->e_phnum; i++) {
    phdr = (Elf32_Phdr *)(buffer + ehdr->e_phoff + i * ehdr->e_phentsize);
    printf("Program Header[%d] =>\n", i);
    printf("  Segment type %d\n", phdr->p_type);
    printf("  Segment file offset 0x%x\n", phdr->p_offset);
    printf("  Segment virtual address 0x%x\n", phdr->p_vaddr);
    printf("  Segment physical address 0x%x\n", phdr->p_paddr);
    printf("  Segment size in file %d\n", phdr->p_filesz);
    printf("  Segment size in memory %d\n", phdr->p_memsz);
    printf("  Segment flags 0x%x\n", phdr->p_flags);
    printf("  Segment alignment 0x%x\n", phdr->p_align);
  }
  printf("\n");

  for(i = 0; i < ehdr->e_shnum; i++) {
    shdr = (Elf32_Shdr *)(buffer + ehdr->e_shoff + i * ehdr->e_shentsize);
    printf("Section Header[%d] =>\n", i);
    if(shdr->sh_type == SHT_NULL) {
      printf("  Empty!\n");
      continue;
    }
    printf("  Section name 0x%x\n", shdr->sh_name);
    printf("  Section type 0x%x\n", shdr->sh_type);
    printf("  Section flags 0x%x\n", shdr->sh_flags);
    printf("  Section virtual addr at execution 0x%x\n", shdr->sh_addr);
    printf("  Section file offset 0x%x\n", shdr->sh_offset);
    printf("  Section size in bytes %d\n", shdr->sh_size);
    printf("  Link to another section 0x%x\n", shdr->sh_link);
    printf("  Additional section information 0x%x\n", shdr->sh_info);
    printf("  Section alignment 0x%x\n", shdr->sh_addralign);
    printf("  Entry size if section holds table %d\n", shdr->sh_entsize);
  }

  return 0;
}
