/*=======================================================================*/
/* MKEBI.C                                                               */
/*                                                                       */
/* Desc: Tool to generate an EPOS bootable image.			 */
/*                                                                       */
/* Parm: <boot image> <os image> <app1> <app2> ...	                 */
/*                                                                       */
/* Auth: Guto			                                         */
/*=======================================================================*/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <fcntl.h>
#include <system/boot.h>
#include <system/setup.h>

#define MASTER_BOOT_ID 0xaa55
#define MASTER_BOOT_OFFSET 510
#define BOOT_STRAP_SECTOR_COUNT_OFFSET 508
#define BOOT_INFO_OFFSET 512
#define FLOPPY_SIZE 1474560

int put_file(int fd_out, char *file);
int put_short(int fd_out, unsigned short num);
int put_long(int fd_out, unsigned long num);
int put_buf(int fd_out, void *buf, int size);
int pad(int fd_out, int size);

int main(int argc, char **argv)
{
    int i, fd_out, image_size;
    char *epos_home;
    char file[256];
    struct stat file_stat;
    Boot_Info bi;
    
    if(argc < 2) {
	fprintf(stderr, "Usage: %s <boot image> <app1> <app2> ...\n", argv[0]);
	return 1;
    }
    
    /* Say hello */
    printf("EPOS bootable image tool\n\n");
    
    /* Get EPOS environment variable */
    epos_home = getenv("EPOS");
    if(epos_home == NULL) {
	fprintf(stderr, "Error: environment variable EPOS not set!\n");
	return 1;
    }

    /* Open destination file (rewrite) */
    fd_out = open(argv[1], O_CREAT | O_TRUNC | O_WRONLY);
    if(fd_out < 0) {
	fprintf(stderr, "Error: can't create boot image file \"%s\"!\n",
		argv[1]);
	return 1;
    }
    else
	printf("Writing image to \"%s\".\n", argv[1]);

    /* Add EPOS */
    image_size = 0;
    printf("Adding EPOS =>\n");

    /* Add the boot strap */
    sprintf(file, "%s/img/boot", epos_home);
    printf("  Boot strap \"%s\" ...", file);
    image_size += put_file(fd_out, file);
    if(image_size != 512) {
	printf(" filed!\n");
	fprintf(stderr, "Boot strap \"%s\" corrupted!\n", file);
	return 1;
    }
    else
	printf(" done.\n");
  
    /* Reserve space for Boot_Info */
    image_size += pad(fd_out, 512);

    /* Add the setup */
    bi.setup_off = image_size - 512;
    sprintf(file, "%s/img/setup", epos_home);
    printf("  Setup \"%s\" ...", file);
    image_size += put_file(fd_out, file);
    printf(" done.\n");

    /* Add the system */
    bi.system_off = image_size - 512;
    sprintf(file, "%s/img/system", epos_home);
    printf("  System \"%s\" ...", file);
    image_size += put_file(fd_out, file);
    printf(" done.\n");

    /* Add the init */
    bi.init_off = image_size - 512;
    sprintf(file, "%s/img/init", epos_home);
    printf("  Init \"%s\" ...", file);
    image_size += put_file(fd_out, file);
    printf(" done.\n");

    /* Add the loader */
    bi.loader_off = image_size - 512;
    if(argc == 3) { /* Single app */
	printf("Single application \"%s\" ...", argv[2]);
	image_size += put_file(fd_out, argv[2]);
    } else {
	sprintf(file, "%s/img/loader", epos_home);
	printf("  Loader \"%s\" ...", file);
	image_size += put_file(fd_out, file);
    }
    printf(" done.\n");

    /* Add applications */
    bi.app_off = image_size - 512;
    if(argc > 3) {
	printf("Adding applications =>");
	for(i = 2; i < argc; i++) {
	    printf("  %s ...", argv[i]);
	    stat(argv[i], &file_stat);
	    image_size += put_long(fd_out, file_stat.st_size);
	    image_size += put_file(fd_out, argv[i]);
	    printf (" done.\n");
	}
	/* Signalize last application by writing 0 to the size of netx app */
	image_size += put_long(fd_out, 0);
    }

    /* Pad the image to the size of a standard floppy */
    pad(fd_out, FLOPPY_SIZE - image_size);

    /* Prepare the Boot Map */
    bi.mem_size = 256 * 1024;
    bi.cpu_type = 6;
    /*  bi.cpu_clock = 266666666; */
    bi.cpu_clock = 300000000;
    bi.node_id = -1; /* get from net */
    bi.n_threads = 8;
    bi.n_tasks = 1;
    bi.img_size = image_size - 512; /* Boot strap is not included */

    /* write the number of sectors to be read */
    if(lseek(fd_out, BOOT_STRAP_SECTOR_COUNT_OFFSET, SEEK_SET) < 0) {
	fprintf(stderr, "Error: can not seek the boot image!\n");
	return 1;
    }
    put_short(fd_out, ((image_size + 511) / 512));

    if(lseek(fd_out, MASTER_BOOT_OFFSET, SEEK_SET) < 0) {
	fprintf(stderr, "Error: can not seek the boot image!\n");
	return 1;
    }
    put_short(fd_out, MASTER_BOOT_ID);

    if(lseek(fd_out, BOOT_INFO_OFFSET, SEEK_SET) < 0) {
	fprintf(stderr, "Error: can not seek the boot image!\n");
	return 1;
    }
    put_buf(fd_out, &bi, sizeof(Boot_Info));


    close(fd_out);

    printf("Image size: %d bytes (%d sectors).\n",
	   image_size, ((image_size + 511) / 512));

    return 0;
}

/*=======================================================================*/
/* PUT_FILE                                                              */
/*                                                                       */
/* Desc: Copy a file to fd0.                                             */
/*                                                                       */
/* Parm: file                                                            */
/*                                                                       */
/* Rtrn: number of bytes written.                                        */
/*                                                                       */
/* Creation date: 10/14/96                                               */
/* Last Updated : 10/14/96     Authors: Aboelha Development Team         */
/*=======================================================================*/
int put_file(int fd_out, char *file)
{
    int fd_in;
    struct stat stat;
    char *buffer;

    fd_in = open(file, O_RDONLY);
    if(fd_in < 0) {
	fprintf(stderr, "Error: can not open %s!\n", file);
	return 0;
    }

    if(fstat(fd_in, &stat) < 0)  {
	fprintf(stderr, "Error: can not stat %s!\n", file);
	return 0;
    }

    buffer = malloc(stat.st_size);
    if(buffer == NULL) {
	fprintf(stderr, "Error: not enough memory!\n");
	return 0;
    }
    memset(buffer, '\1', stat.st_size);

    if(read(fd_in, buffer, stat.st_size) < 0) {
	fprintf(stderr, "Error: can not read from %s!\n", file);
	return 0;
    }

    if(write(fd_out, buffer, stat.st_size) < 0) {
	fprintf(stderr, "Error: can not write to the boot image!\n");
	return 0;
    }

    free(buffer);
    
    close(fd_in);
  
    return(stat.st_size);
}

/*=======================================================================*/
/* WRITE_NUMBER                                                          */
/*                                                                       */
/* Desc: Write a number at a position in fd0.                            */
/*                                                                       */
/* Parm: num     -> number to write                                      */
/*       offset  -> offset in fd0                                        */
/*                                                                       */
/* Rtrn: nothing.                                                        */
/*                                                                       */
/* Creation date: 10/14/96                                               */
/* Last Updated : 10/14/96     Authors: Aboelha Development Team         */
/*=======================================================================*/
int put_short(int fd_out, unsigned short num)
{
    if(write(fd_out, &num, sizeof(short)) < 0) {
	fprintf(stderr, "Error: can not wirte to file!\n");
	return 0;
    }

    return sizeof(short);
}

int put_long(int fd_out, unsigned long num)
{
    if(write(fd_out, &num, sizeof(long)) < 0) {
	fprintf(stderr, "Error: can not wirte to file!\n");
	return 0;
    }

    return sizeof(long);
}

int put_buf(int fd_out, void *buf, int size)
{
    int written;

    if((written = write(fd_out, buf, size)) < 0) {
	fprintf(stderr, "Error: can not wirte to file!\n");
	return 0;
    }

    return written;
}

int pad(int fd_out, int size)
{
    char * buffer;

    buffer = malloc(size);
    if(buffer == NULL) {
	fprintf(stderr, "Error: not enough memory!\n");
	return 0;
    }
    memset(buffer, '\1', size);

    if(write(fd_out, buffer, size) < 0) {
	fprintf(stderr, "Error: can not write to the boot image!\n");
	return 0;
    }

    return size;
}




















