// EPOS DescriptorsManager Implementation
//
// Author: laureano
// Documentation: $EPOS/doc/filesystem			Date: 28 Jun 2004

#include <descriptorsmanager.h>
#include <filesysteminformation.h>
#include <mediator.h>

#include <iostream>
using namespace std;

__BEGIN_SYS
__BEGIN_IMP

DescriptorsManager* DescriptorsManager::dm;

DescriptorsManager::DescriptorsManager()
{
  cout << "DescriptorsManager()\n";
}


DescriptorsManager::~DescriptorsManager()
{
  cout << "~DescriptorsManager()\n";
}


DescriptorsManager* DescriptorsManager::getDescriptorsManager()
{
  if (dm == 0)
    dm = new DescriptorsManager();
  
  return dm;
}

void DescriptorsManager::setAllDescriptorsAsFree()
{
  FileSystemInformation *fsi = FileSystemInformation::getFileSystemInformation();
  unsigned int start = fsi->getWhereBitsMapStart();  
  unsigned int end = start + fsi->getNumberDescriptorsForBitsMap(); 
  Mediator* mediator = Mediator::getMediator();
  Address blockAddress = start / DESCRIPTORS_FOR_BLOCK;
  Block *block = mediator->readBlock(blockAddress);  

  Address newBlockAddress;
  for (unsigned int i = start; i < end; i++)
  {
    //Verify if it's end of block
    newBlockAddress = i / DESCRIPTORS_FOR_BLOCK;
    if (blockAddress != newBlockAddress)
    {
      //Get the nextBlock
      mediator->writeBlock(block, blockAddress);
      delete block;
    
      blockAddress = newBlockAddress;
      block = mediator->readBlock(blockAddress);
    }  
  
    //Release all descriptors (states) stored in this descriptor
    for (unsigned int j = 0; j < sizeof(FileDescriptor); j++)
      ((char *)&block->descriptor[i % DESCRIPTORS_FOR_BLOCK])[j] = 0x00;
  }
  
  mediator->writeBlock(block, blockAddress);
  delete block;
}


FileDescriptor* DescriptorsManager::getDescriptor(Address numberOfDescriptor)
{
  FileSystemInformation *fsi = FileSystemInformation::getFileSystemInformation();
  numberOfDescriptor += fsi->getWhereDescriptorsStart();    
  Address blockAddress = numberOfDescriptor / DESCRIPTORS_FOR_BLOCK;  
  Mediator* mediator = Mediator::getMediator();
  Block *block = mediator->readBlock(blockAddress);  
  FileDescriptor *fd = new FileDescriptor;
  Address indexOfDescriptor = numberOfDescriptor % DESCRIPTORS_FOR_BLOCK;

  //copy descriptor
  copy(&block->descriptor[indexOfDescriptor], fd, sizeof(FileDescriptor));

  delete block;
  return fd;
}
    

bool DescriptorsManager::writeDescriptor(FileDescriptor* desc, Address numberOfDes)
{
  FileSystemInformation *fsi = FileSystemInformation::getFileSystemInformation();
  numberOfDes += fsi->getWhereDescriptorsStart();    
  Address blockAddress = numberOfDes / DESCRIPTORS_FOR_BLOCK;  
  Mediator* mediator = Mediator::getMediator();
  Block *block = mediator->readBlock(blockAddress);  
  Address indexOfDescriptor = numberOfDes % DESCRIPTORS_FOR_BLOCK;

  //copy descriptor
  copy(desc, &block->descriptor[indexOfDescriptor], sizeof(FileDescriptor));

  mediator->writeBlock(block, blockAddress);
  delete block;
  
  return true;
}
    

Address DescriptorsManager::getFreeDescriptor()
{  
  cout << "getFreeDescriptor()\n";
  FileSystemInformation *fsi = FileSystemInformation::getFileSystemInformation();
  unsigned int start = 0;  
  unsigned int end = fsi->getNumberDescriptorsForBitsMap(); 
  Address blockAddress = fsi->getWhereBitsMapStart() / DESCRIPTORS_FOR_BLOCK;

  Mediator* mediator = Mediator::getMediator();
  Block *block = mediator->readBlock(blockAddress);  
   

  Address newBlockAddress;
  for (unsigned int i = start; i < end; i++)
  {
    //Verify if it's end of block
    newBlockAddress = (i / 8 + sizeof(FileDescriptor)*fsi->getWhereBitsMapStart()) / BLOCK_SIZE;
    if (blockAddress != newBlockAddress)
    {
      //Get the nextBlock
      delete block;
    
      blockAddress = newBlockAddress;
      block = mediator->readBlock(blockAddress);
    }
    cout << "   blockAddress = " << blockAddress << endl;
  
    //Release the descriptors
    unsigned char index = i / 8 + sizeof(FileDescriptor)*fsi->getWhereBitsMapStart();
    cout << "   index = " << (int)index << endl;

    if (block->byte[index] != 0xFF)
    {
      //Catch the index of new descriptor
      unsigned char j;
      for (j = 0; block->byte[index] & (0x80 >> j); j++);
      
      cout << "    block->byte[index] Old value = " << (int)block->byte[index] << endl; 
      block->byte[index] |= (0x80 >> j); //set as used
      cout << "    block->byte[index] New value = " << (int)block->byte[index] << endl;  

      mediator->writeBlock(block, blockAddress);
      delete block;
      
      return (i*8 + j); //litle endian
    }
  }
  
  delete block;
  return false;
}
    
FileDescriptor* DescriptorsManager::getFreeDescriptor(Address numberOfDescriptor)
{
  cout << "getFreeDescriptor(" << numberOfDescriptor << ")\n";
  FileSystemInformation *fsi = FileSystemInformation::getFileSystemInformation();
  Address blockAddress = (numberOfDescriptor / 8 + fsi->getWhereBitsMapStart()*sizeof(FileDescriptor)) / BLOCK_SIZE;
  cout << "   blockAddress = " << blockAddress << endl;

  Mediator* mediator = Mediator::getMediator();
  Block *block = mediator->readBlock(blockAddress);

  Address index = numberOfDescriptor / 8 + sizeof(FileDescriptor)*fsi->getWhereBitsMapStart();
  cout << "   index = " << (int)index << endl;

  //Erase bit, releasing the descriptor;s
  unsigned char mask = 0x80 >> (numberOfDescriptor % 8); //bit inside byte
  if ((block->byte[index] & mask) != 0)
    return 0;

  cout << "    block->byte[index] Old value = " << (int)block->byte[index] << endl;
  block->byte[index] |= mask;
  cout << "    block->byte[index] New value = " << (int)block->byte[index] << endl;


  mediator->writeBlock(block, blockAddress);
  delete block;

  return getDescriptor(numberOfDescriptor);
}


void DescriptorsManager::releaseDescriptor(Address numberOfDescriptor)
{
  FileSystemInformation *fsi = FileSystemInformation::getFileSystemInformation();    
  Address blockAddress = (numberOfDescriptor/8 + fsi->getWhereBitsMapStart()*sizeof(FileDescriptor)) / BLOCK_SIZE;

  Mediator* mediator = Mediator::getMediator();
  Block *block = mediator->readBlock(blockAddress);  
 
  Address indexOfByte = (numberOfDescriptor/8 +sizeof(FileDescriptor)*fsi->getWhereBitsMapStart()) % BLOCK_SIZE;

  //Erase bit, releasing the descriptor;s
  unsigned char mask = 0x80 >> (numberOfDescriptor % 8); //bit in byte
  block->byte[indexOfByte] &= ~mask;

  mediator->writeBlock(block, blockAddress);
  delete block;
}

__END_IMP
__END_SYS
